import { PlayerType } from 'pulsar';
import type { FC } from 'react';
import { graphql, useFragment } from 'react-relay/hooks';
import { useEffectOnce } from 'tachyon-utils';
import { useRefetchPlaybackAccessToken } from '../../useRefetchPlaybackAccessToken';
import { VodPlayerBase, vodPlayerVideoFragment } from '../VodPlayerBase';
import type { VodPlayerBaseProps } from '../VodPlayerBase';
import type { VodPreviewPlayerWrapper_previewToken$key } from './__generated__/VodPreviewPlayerWrapper_previewToken.graphql';

export type {
  VodPreviewPlayerWrapper_previewToken$key,
  VodPreviewPlayerWrapper_previewToken,
} from './__generated__/VodPreviewPlayerWrapper_previewToken.graphql';

const vodPreviewTokenFragment = graphql`
  fragment VodPreviewPlayerWrapper_previewToken on Query
  @refetchable(queryName: "VodPreviewPlayerWrapper_RefetchPreviewToken") {
    user(login: $login) @skip(if: $skipPlayToken) {
      videos(first: 1) {
        edges {
          node {
            playbackAccessToken(
              params: { platform: $platform, playerType: $playerType }
            ) {
              signature
              value
            }
          }
        }
      }
    }
  }
`;

const vodPreviewTokenRefetchQuery = graphql`
  query VodPreviewPlayerWrapper_Query(
    $login: String!
    $platform: String!
    $playerType: String!
    $skipPlayToken: Boolean!
  ) {
    ...VodPreviewPlayerWrapper_previewToken
  }
`;

export type VodPreviewPlayerWrapperProps = Omit<
  VodPlayerBaseProps,
  'playbackAccessToken' | 'playerType'
> & {
  vodPreviewToken: VodPreviewPlayerWrapper_previewToken$key;
};

/**
 * Wrapper responsible for refetching the playback access token fragment
 * on the ChannelHomepage query. Used for the embedded preview player.
 */
// istanbul ignore next: trivial
export const VodPreviewPlayerWrapper: FC<VodPreviewPlayerWrapperProps> = ({
  vodPreviewToken: vodPreviewTokenRef,
  ...props
}) => {
  const video = useFragment(vodPlayerVideoFragment, props.video);

  const playerType = PlayerType.Max480pNoAds;
  const { refetchToken, token: vodPreviewToken } =
    useRefetchPlaybackAccessToken({
      playerType,
      refetchQuery: vodPreviewTokenRefetchQuery,
      tokenFragment: vodPreviewTokenFragment,
      tokenRef: vodPreviewTokenRef,
      videoId: video.id,
    });

  useEffectOnce(() => {
    if (!vodPreviewToken.user?.videos) {
      refetchToken();
    }
  });

  const edges = vodPreviewToken.user?.videos?.edges ?? [];
  const playbackAccessToken = edges[0]?.node?.playbackAccessToken ?? null;

  return (
    <VodPlayerBase
      playbackAccessToken={playbackAccessToken}
      playerType={playerType}
      {...props}
    />
  );
};

VodPreviewPlayerWrapper.displayName = 'VodPreviewPlayerWrapper';
