import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { formatTotalViewCount } from 'tachyon-more-ui';
import { formatVideoLength } from 'tachyon-utils';
import { Layout, MediaCardStat, Position } from 'twitch-core-ui';
import type { VodPlayerOverlay_video } from './__generated__/VodPlayerOverlay_video.graphql';

type VodPlayerOverlayBaseProps = {
  video: VodPlayerOverlay_video;
};

// istanbul ignore next: trivial
/**
 * Component that renders additional stuff on top of a playing stream,
 * indicating information like stream type and viewer count.
 */
export const VodPlayerOverlayBase: FC<VodPlayerOverlayBaseProps> = ({
  video,
}) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();
  return (
    <>
      {video.createdAt && (
        <Layout
          attachLeft
          attachTop
          margin={{ left: 1, top: 1 }}
          position={Position.Absolute}
        >
          <MediaCardStat>
            {formatPastRelativeDate(new Date(video.createdAt))}
          </MediaCardStat>
        </Layout>
      )}
      {video.viewCount && (
        <Layout
          attachBottom
          attachLeft
          margin={{ bottom: 1, left: 1 }}
          position={Position.Absolute}
        >
          <MediaCardStat>
            {formatTotalViewCount(
              formatMessage,
              video.viewCount,
              formatNumberShort(video.viewCount, {
                maximumSignificantDigits: 2,
              }),
            )}
          </MediaCardStat>
        </Layout>
      )}
      {video.lengthSeconds && (
        <Layout
          attachBottom
          attachRight
          margin={{ bottom: 1, right: 1 }}
          position={Position.Absolute}
        >
          <MediaCardStat>
            {formatVideoLength(video.lengthSeconds)}
          </MediaCardStat>
        </Layout>
      )}
    </>
  );
};
VodPlayerOverlayBase.displayName = 'VodPlayerOverlayBase';

// istanbul ignore next: trivial
export const VodPlayerOverlay = createFragmentContainer(VodPlayerOverlayBase, {
  video: graphql`
    fragment VodPlayerOverlay_video on Video {
      createdAt
      lengthSeconds
      viewCount
    }
  `,
});

VodPlayerOverlay.displayName = 'VodPlayerOverlay';
