import { PlayerControllerRoot, VodPlaybackRestrictionsRoot } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { ClientOnly, getChannelThumbnailSrcAndSrcSet } from 'tachyon-more-ui';
import { CoreImage } from 'twitch-core-ui';
import { PreviewRestriction } from '../Restrictions';
import type { VodPreviewPlayerWrapperProps } from '../VodPlayer';
import { VodPreviewPlayerWrapper } from '../VodPlayer';
import { VodPlayerOverlay } from './VodPlayerOverlay';
import type { VodPreviewPlayer_currentUser } from './__generated__/VodPreviewPlayer_currentUser.graphql';
import type { VodPreviewPlayer_video } from './__generated__/VodPreviewPlayer_video.graphql';

export type VodPreviewPlayerBaseProps = Pick<
  VodPreviewPlayerWrapperProps,
  'vodPreviewToken'
> & {
  currentUser: VodPreviewPlayer_currentUser | null;
  video: VodPreviewPlayer_video;
};

/**
 * Component intended to be embedded within a page that plays a vod.
 * Will always be muted, ignores captions, and ignores content restrictions.
 */
export const VodPreviewPlayerBase: FC<VodPreviewPlayerBaseProps> = ({
  currentUser,
  video,
  vodPreviewToken,
}) => {
  const {
    common: { platform },
  } = useStaticEnvironment();

  // Playback on Switch results in fullscreen take over by a native UI, so show a preview image instead
  if (platform === Platform.Switch) {
    return (
      <CoreImage
        alt=""
        {...getChannelThumbnailSrcAndSrcSet(video.previewThumbnailURL)}
      />
    );
  }

  return (
    <PreviewRestriction>
      <ClientOnly>
        <VodPreviewPlayerWrapper
          autoPlay
          currentUser={currentUser}
          muted
          video={video}
          vodPreviewToken={vodPreviewToken}
        />
      </ClientOnly>
      <VodPlayerOverlay video={video} />
    </PreviewRestriction>
  );
};
VodPreviewPlayerBase.displayName = 'VodPreviewPlayerBase';

// istanbul ignore next: trivial
export const VodPreviewPlayer = createFragmentContainer(
  (props: VodPreviewPlayerBaseProps) => {
    return (
      <PlayerControllerRoot>
        <VodPlaybackRestrictionsRoot video={props.video}>
          <VodPreviewPlayerBase {...props} />
        </VodPlaybackRestrictionsRoot>
      </PlayerControllerRoot>
    );
  },
  {
    currentUser: graphql`
      fragment VodPreviewPlayer_currentUser on User {
        ...VodPlayerBase_currentUser
      }
    `,
    video: graphql`
      fragment VodPreviewPlayer_video on Video {
        ...VodPlayerBase_video
        ...VodPlayerOverlay_video
        ...VodRestriction_video @relay(mask: false)
        previewThumbnailURL
      }
    `,
  },
);

VodPreviewPlayer.displayName = 'VodPreviewPlayer';
