import { act, renderHook } from '@testing-library/react-hooks';
import { internet } from 'faker';
import { PlayerType, usePlaybackRestrictions } from 'pulsar';
import { useRefetchableFragmentWithoutSuspense } from 'tachyon-relay';
import { streamTokenFragment } from '../StreamPlayer';
import type { StreamPlayer_token$key } from '../StreamPlayer';
import type { useRefetchPlaybackAccessTokenChannelOpts } from './types';
import { useRefetchPlaybackAccessToken } from '.';

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlaybackRestrictions: jest.fn(() => null),
}));

const mockUsePlaybackRestrictions = usePlaybackRestrictions as jest.Mock;

jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  useRefetchableFragmentWithoutSuspense: jest.fn(() => [{}, () => null]),
}));
const mockUseRefetchableFragmentWithoutSuspense =
  useRefetchableFragmentWithoutSuspense as jest.Mock;

const streamTokenRef: StreamPlayer_token$key = {
  ' $fragmentRefs': {
    StreamPlayer_token: true,
  },
};

type OnComplete = (error?: Error) => void;

describe(useRefetchPlaybackAccessToken, () => {
  function setup() {
    return renderHook(() =>
      useRefetchPlaybackAccessToken({
        login: internet.userName(),
        playerType: PlayerType.Pulsar,
        tokenFragment: streamTokenFragment,
        tokenRef: streamTokenRef,
      } as useRefetchPlaybackAccessTokenChannelOpts),
    );
  }

  it('adds a restriction when an error occurs', () => {
    const mockError = new Error('mock');
    const mockAddRestriction = jest.fn();
    mockUsePlaybackRestrictions.mockImplementation(() => ({
      addRestriction: mockAddRestriction,
    }));
    mockUseRefetchableFragmentWithoutSuspense.mockImplementationOnce(() => [
      null,
      (_: any, { onComplete }: { onComplete: OnComplete }) => {
        onComplete(mockError);
      },
    ]);
    const { result } = setup();
    act(() => {
      result.current.refetchToken();
    });
    expect(mockAddRestriction).toHaveBeenCalledWith('playback-error');
  });

  it('does not add a restriction when no error exists', () => {
    const mockAddRestriction = jest.fn();
    mockUsePlaybackRestrictions.mockImplementationOnce(() => ({
      addRestriction: mockAddRestriction,
    }));
    mockUseRefetchableFragmentWithoutSuspense.mockImplementationOnce(() => [
      null,
      (_: any, { onComplete }: { onComplete: OnComplete }) => {
        onComplete();
      },
    ]);
    const { result } = setup();
    act(() => {
      result.current.refetchToken();
    });
    expect(mockAddRestriction).not.toHaveBeenCalled();
  });
});
