import type { MarginValue } from 'twitch-core-ui';

/**
 * This value is set to be slow enough that:
 * - App performance is not completely degredaded from the user holding down inputs
 * - The app feels responsive to the user, 300ms is when things start to feel noticeably slow to the user
 */
export const KEY_NAV_THROTTLE_WAIT_MS = 100;
export const KEY_NAV_THROTTLE_WAIT_S = KEY_NAV_THROTTLE_WAIT_MS / 1000;
export const CARD_ANIMATION_TIMING_S = KEY_NAV_THROTTLE_WAIT_S;

/**
 * This input is really spammy, so we set it really high.
 */
export const SCROLL_NAV_THROTTLE_WAIT_MS = 1000;

/**
 * The amount of time to debounce UI changes related to a new Card being focused.
 */
export const CARD_FOCUS_META_DEBOUNCE_MS = 300;

/**
 * Delay until spinner shows.
 *
 * nngroup.com/articles/website-response-times/
 */
export const LOADING_SHOW_DELAY_MS = 1000;

/**
 * Time to wait before declaring there has been no recent user input.
 */
export const RECENT_USER_INPUT_TIMEOUT_MS = 5000;

/**
 * Ensures that content remain in the "Safe Area" to avoid being clipped by
 * overscan.
 *
 * LG suggests 20px.
 * More details: http://webostv.developer.lge.com/design/style/overscan/.
 *
 * Currently set at a much larger value per design's preference.
 */
// can't divide here because of core-ui types
export const OVERSCAN_PADDING_REM = 4;

/**
 * Cards of all types are the same height to simplify vertical scrolling with
 * virtualization.
 */
export const CARD_HEIGHT_REM = 20;

/**
 * Value chosen by manually inspecting how much margin is needed to ensure
 * the upper virtualization buffer row of cards is completely hidden.
 */
export const GRID_CARD_BOTTOM_MARGIN_REM: Exclude<MarginValue, 'auto'> = 3;

/**
 * Total card height allotted to each card when used in a Grid.
 */
export const GRID_CARD_HEIGHT_REM =
  CARD_HEIGHT_REM + GRID_CARD_BOTTOM_MARGIN_REM;

export type CardConfig = {
  // the number of cards that fit in a single row at their current size
  cardsPerRow: number;
  horizontalPaddingRem: number;
  widthRem: number;
};

// 1920px - 144px (OVERSCAN_PADDING_REM 4rem * 18px * 2) = 1776px shelf width
// 1776px / 4 card per row = 444px = Stream Card Width
// 1776px / 8 card per row = 222px = Category Card Width

export const CATEGORY_CARD_CONFIG: CardConfig = {
  cardsPerRow: 8,
  horizontalPaddingRem: 1,
  widthRem: 12.333333,
};

export const WATCHABLE_CARD_CONFIG: CardConfig = {
  cardsPerRow: 4,
  horizontalPaddingRem: 1,
  widthRem: 24.666666667,
};

// Size of the avatar on a stream card
// Also used to override media card image flex-basis for Chrome 38
export const STREAM_CARD_AVATAR_SIZE_REM = 4;

// Size of the game box art on a clip or vod card
export const VIDEO_CARD_BOX_ART_SIZE_REM = 55;

// Size of the offline channel image
export const OFFLINE_CHANNEL_AVATAR_SIZE_REM = 12.5;

// Needs to have same dimensions as a Watchable Card so it can be
// displayed in the same shelf on the Search Page
export const OFFLINE_CHANNEL_CARD_CONFIG: CardConfig = WATCHABLE_CARD_CONFIG;

export const TOP_NAV_HEIGHT_REM = 6 + OVERSCAN_PADDING_REM;

export const PRIMARY_QR_CODE_SIZE_REM = 42;

export const CARD_ANIMATION = '200ms ease-in-out';
