import { graphql } from 'react-relay/hooks';
import { useIntl } from 'tachyon-intl';
import { isValidObject } from 'tachyon-relay';
import type { OmitRefType } from 'tachyon-type-library';
import { useStarshotMutation } from '../useStarshotMutation';
import type { useFollow_FollowMutationVariables } from './__generated__/useFollow_FollowMutation.graphql';
import type { useFollow_UnfollowMutationVariables } from './__generated__/useFollow_UnfollowMutation.graphql';
import type { useFollow_targetChannel } from './__generated__/useFollow_targetChannel.graphql';

export type FollowTargetChannel = OmitRefType<useFollow_targetChannel>;

interface UseFollow {
  follow: () => void;
  following: boolean;
  unfollow: () => void;
}

export function useFollow(channel: FollowTargetChannel | null): UseFollow {
  const { formatMessage } = useIntl();

  const commitFollow = useStarshotMutation<useFollow_FollowMutationVariables>({
    errorMessage: formatMessage(
      'Failed to follow. Please try again.',
      'useFollow',
    ),
    mutation: followMutation,
  });

  const commitUnfollow =
    useStarshotMutation<useFollow_UnfollowMutationVariables>({
      errorMessage: formatMessage(
        'Failed to unfollow. Please try again.',
        'useFollow',
      ),
      mutation: unfollowMutation,
    });

  if (!isValidObject(channel)) {
    return {
      follow: () => undefined,
      following: false,
      unfollow: () => undefined,
    };
  }

  return {
    follow: () => {
      commitFollow({
        input: {
          disableNotifications: true,
          targetID: channel.id,
        },
      });
    },
    following: !!channel.self?.follower?.followedAt,
    unfollow: () => {
      commitUnfollow({
        input: {
          targetID: channel.id,
        },
      });
    },
  };
}

const followMutation = graphql`
  mutation useFollow_FollowMutation($input: FollowUserInput!) {
    followUser(input: $input) {
      follow {
        user {
          id
          ...useFollow_targetChannel @relay(mask: false)
        }
      }
    }
  }
`;

const unfollowMutation = graphql`
  mutation useFollow_UnfollowMutation($input: UnfollowUserInput!) {
    unfollowUser(input: $input) {
      follow {
        user {
          id
          ...useFollow_targetChannel @relay(mask: false)
        }
      }
    }
  }
`;

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment useFollow_targetChannel on User {
    id
    self {
      follower {
        followedAt
      }
    }
  }
`;
