import { manifestHasMinimumQualities } from 'pulsar';
import { useMemo, useRef } from 'react';

const SWITCH_PLAYABLE_QUALITIES = [720, 480, 360];
const falsePromise = Promise.resolve(false);
const truePromise = Promise.resolve(true);

type SetManifestUrl = (url: string) => void;
export type IsContentPlayable = () => Promise<boolean>;

type UseIsContentPlayableOnSwitch = {
  isContentPlayable: IsContentPlayable;
  setManifestUrl: SetManifestUrl;
};

export const useIsContentPlayableOnSwitch =
  (): UseIsContentPlayableOnSwitch => {
    const isContentPlayable = useRef<Promise<boolean>>(falsePromise);

    return useMemo(() => {
      const setManifestUrl: SetManifestUrl = (url) => {
        try {
          isContentPlayable.current = manifestHasMinimumQualities(
            url,
            SWITCH_PLAYABLE_QUALITIES,
          );
        } catch (error) {
          // If we're unable to determine whether the content is playable
          // default to false to avoid potential playback errors. Users
          // can try again by navigating away from and back to the page.
          isContentPlayable.current = falsePromise;
        }
      };
      return {
        isContentPlayable: () => isContentPlayable.current,
        setManifestUrl,
      };
    }, []);
  };

type SetClipManifestUrl = (url: string | null) => void;

type UseIsClipPlayableOnSwitch = {
  isContentPlayable: IsContentPlayable;
  setManifestUrl: SetClipManifestUrl;
};

export const useIsClipPlayableOnSwitch = (): UseIsClipPlayableOnSwitch => {
  const isContentPlayable = useRef<Promise<boolean>>(falsePromise);

  return useMemo(() => {
    const setManifestUrl: SetClipManifestUrl = (url) => {
      isContentPlayable.current = url === null ? falsePromise : truePromise;
    };
    return {
      isContentPlayable: () => isContentPlayable.current,
      setManifestUrl,
    };
  }, []);
};
