import { act, renderHook } from '@testing-library/react-hooks';
import { manifestHasMinimumQualities } from 'pulsar';
import { useIsClipPlayableOnSwitch, useIsContentPlayableOnSwitch } from '.';

jest.mock('pulsar', () => ({
  manifestHasMinimumQualities: jest.fn(),
}));

const mockManifestHasMinimumQualities =
  manifestHasMinimumQualities as jest.Mock;

describe(useIsClipPlayableOnSwitch, () => {
  it('updates the return value based on the manifest url provided', async () => {
    const { result } = renderHook(() => useIsClipPlayableOnSwitch());
    expect(await result.current.isContentPlayable()).toBe(false);

    act(() => {
      result.current.setManifestUrl('test');
    });
    expect(await result.current.isContentPlayable()).toBe(true);
    act(() => {
      result.current.setManifestUrl(null);
    });
    expect(await result.current.isContentPlayable()).toBe(false);
  });
});

describe(useIsContentPlayableOnSwitch, () => {
  it('returns false when manifest is missing requisite qualities', async () => {
    mockManifestHasMinimumQualities.mockReturnValueOnce(false);
    const { result } = renderHook(() => useIsContentPlayableOnSwitch());
    expect(await result.current.isContentPlayable()).toBe(false);

    act(() => {
      result.current.setManifestUrl('test');
    });
    expect(await result.current.isContentPlayable()).toBe(false);
  });

  it('returns false when there is an error fetching the manifest', async () => {
    mockManifestHasMinimumQualities.mockImplementationOnce(() => {
      throw new Error('err');
    });
    const { result } = renderHook(() => useIsContentPlayableOnSwitch());
    expect(await result.current.isContentPlayable()).toBe(false);

    act(() => {
      result.current.setManifestUrl('test');
    });
    expect(await result.current.isContentPlayable()).toBe(false);
  });

  it('returns true when manifest has minimum qualities', async () => {
    mockManifestHasMinimumQualities.mockReturnValueOnce(true);
    const { result } = renderHook(() => useIsContentPlayableOnSwitch());
    expect(await result.current.isContentPlayable()).toBe(false);

    act(() => {
      result.current.setManifestUrl('test');
    });
    expect(await result.current.isContentPlayable()).toBe(true);
  });
});
