import { renderHook } from '@testing-library/react-hooks';
import type { SearchHistory } from '.';
import { SEARCH_HISTORY_LENGTH, useSearchHistory } from '.';

let mockSearchHistory: SearchHistory = [];
const mockSetStorage = jest.fn();

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  useStorage: jest.fn(() => [mockSearchHistory, mockSetStorage]),
}));

describe(useSearchHistory, () => {
  beforeEach(() => {
    mockSearchHistory = [];
    mockSetStorage.mockReset();
  });

  it('clears storage', () => {
    const { result } = renderHook(() => useSearchHistory());
    result.current.clearHistory();

    expect(mockSetStorage).toHaveBeenCalledTimes(1);
    expect(mockSetStorage).toHaveBeenCalledWith([]);
  });

  describe('when storage contains history', () => {
    beforeEach(() => {
      mockSearchHistory = ['therealderekt'];
    });

    it('returns storage results', () => {
      const { result } = renderHook(() => useSearchHistory());
      expect(result.current.history).toEqual(mockSearchHistory);
    });

    it('pushing a term adds it to the front of the history list', () => {
      const { rerender, result } = renderHook(() => useSearchHistory());
      result.current.pushTerm('newTerm');
      rerender();

      expect(mockSetStorage).toHaveBeenCalledTimes(1);
      expect(mockSetStorage).toHaveBeenCalledWith(['newTerm', 'therealderekt']);
    });

    it(`adds new term and constrains the list to ${SEARCH_HISTORY_LENGTH} results`, () => {
      mockSearchHistory = Array.from({ length: SEARCH_HISTORY_LENGTH }).map(
        (_, i) => `term-${i}`,
      );

      const { rerender, result } = renderHook(() => useSearchHistory());
      result.current.pushTerm('newTerm');
      rerender();

      expect(mockSetStorage).toHaveBeenCalledTimes(1);
      expect(mockSetStorage).toHaveBeenCalledWith([
        'newTerm',
        'term-0',
        'term-1',
        'term-2',
        'term-3',
        'term-4',
        'term-5',
        'term-6',
        'term-7',
        'term-8',
      ]);
    });

    it(`bumps an existing term to the top of the list`, () => {
      mockSearchHistory = ['a', 'b', 'c'];

      const { rerender, result } = renderHook(() => useSearchHistory());
      result.current.pushTerm('b');
      rerender();

      expect(mockSetStorage).toHaveBeenCalledTimes(1);
      expect(mockSetStorage).toHaveBeenCalledWith(['b', 'a', 'c']);
    });
  });

  describe('when storage has no history', () => {
    beforeEach(() => {
      mockSearchHistory = [];
    });

    it('adds an entry to an empty history', () => {
      const { rerender, result } = renderHook(() => useSearchHistory());
      result.current.pushTerm('newTerm');
      rerender();

      expect(mockSetStorage).toHaveBeenCalledTimes(1);
      expect(mockSetStorage).toHaveBeenCalledWith(['newTerm']);
    });
  });
});
