import { useMutation } from 'react-relay/hooks';
import type { MutationConfig, MutationParameters } from 'tachyon-relay';
import type { ForceExplicitGenericsObject } from 'tachyon-type-library';
import type { Simplify } from 'type-fest';
import { useStarshotNotification } from './useStarshotNotification';

type StarshotConfig<MutationVars> = Simplify<
  // Pick for now to prevent proliferation of one-off extended patterns
  Pick<MutationConfig<MutationVars>, 'mutation'> & {
    /**
     * Providing an errorMessage will trigger a toast to appear if this mutation fails.
     * You can opt into silent failures by passing null instead.
     */
    errorMessage: string | null;
  }
>;

// istanbul ignore next: high cost, low value
export function useStarshotMutation<
  // mutations don't carry a concept of variables shape so we force explicit annotation
  MutationVars extends {} = ForceExplicitGenericsObject,
>({
  errorMessage,
  mutation,
}: StarshotConfig<MutationVars>): (variables: MutationVars) => void {
  const { publish } = useStarshotNotification();
  const [commit, inFlight] =
    useMutation<MutationParameters<MutationVars>>(mutation);

  const onError = errorMessage
    ? () => {
        publish({
          autoCloseMs: 5000,
          meta: { displayType: 'snack-bar', message: errorMessage },
          type: 'error',
        });
      }
    : undefined;

  return (variables: MutationVars) => {
    if (inFlight) {
      return;
    }

    commit({
      onError,
      variables,
    });
  };
}
