import { usePlayerController } from 'pulsar';
import { useEffect, useMemo } from 'react';
import type { SetOptional } from 'type-fest';
import type { UseUpdateUserViewedVideoProps } from './useUpdateUserViewedVideo';
import { useUpdateUserViewedVideo } from './useUpdateUserViewedVideo';

export type UseUpdateVideoPlaybackPositionProps = Omit<
  SetOptional<UseUpdateUserViewedVideoProps, 'userID' | 'videoID'>,
  'position'
>;

// We persist our last video playback position every 30 seconds
export const UPDATE_INTERVAL_S = 30 * 1000;

export function useUpdateVideoPlaybackPosition({
  // null-coalesce to empty-string for ergonomics but still representing no user
  userID = '',
  videoID = '',
  videoType,
}: UseUpdateVideoPlaybackPositionProps): void {
  const playerController = usePlayerController();

  const mutationConfig = useMemo(
    () => ({
      userID,
      videoID,
      videoType,
    }),
    [userID, videoID, videoType],
  );
  const mutation = useUpdateUserViewedVideo(mutationConfig);

  useEffect(() => {
    if (!userID || !videoID || !playerController) {
      return;
    }
    const interval = window.setInterval(() => {
      mutation({
        // Rounding here because player controller returns a floating point number
        // but GQL expects an int32
        position: Math.round(playerController.getPosition()),
      });
    }, UPDATE_INTERVAL_S);
    return () => {
      window.clearInterval(interval);
    };
  }, [userID, mutation, playerController, videoID]);
}
