import { renderHook } from '@testing-library/react-hooks';
import { datatype } from 'faker';
import { mockPlayerController, usePlayerController } from 'pulsar';
import { act } from 'react-dom/test-utils';
import { useUpdateUserViewedVideo } from './useUpdateUserViewedVideo';
import type {
  VideoType,
  useUpdateUserViewedVideoMutationVariables,
} from './useUpdateUserViewedVideo/__generated__/useUpdateUserViewedVideoMutation.graphql';
import type { UseUpdateVideoPlaybackPositionProps } from '.';
import { UPDATE_INTERVAL_S, useUpdateVideoPlaybackPosition } from '.';

jest.mock('../useStarshotMutation', () => ({
  useStarshotMutation: jest.fn(),
}));

jest.mock('./useUpdateUserViewedVideo', () => ({
  useUpdateUserViewedVideo: jest.fn(),
}));
const mockUseUpdateUserViewedVideo = useUpdateUserViewedVideo as jest.Mock;

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  usePlayerController: jest.fn(() => null),
}));
const mockUsePlayerController = usePlayerController as jest.Mock;

const validVideoProps = {
  userID: datatype.uuid(),
  videoID: datatype.uuid(),
  videoType: 'LIVE' as VideoType,
};

const invalidVideoProps = {
  videoType: 'LIVE' as VideoType,
};

describe(useUpdateVideoPlaybackPosition, () => {
  function setup(
    variables: UseUpdateVideoPlaybackPositionProps = {
      videoType: 'LIVE',
    },
  ) {
    return renderHook(
      (props) => {
        useUpdateVideoPlaybackPosition(props);
      },
      {
        initialProps: variables,
      },
    );
  }

  it('passes appropriate props to the mutation hook', () => {
    setup(validVideoProps);

    expect(mockUseUpdateUserViewedVideo).toHaveBeenCalledWith(validVideoProps);
    expect(mockUsePlayerController).toHaveBeenCalled();
  });

  it('calls the function returned by the mutation hook when the video data is valid', () => {
    // Mock player controller position so we know what to expect our mutation to have been called with
    const mockPosition = 1234;
    const mockedPlayerController = mockPlayerController({
      getPosition: () => mockPosition,
    });
    mockUsePlayerController.mockReturnValue(mockedPlayerController);

    // Set up a mock mutation function to be returned by the main mutation hook
    const mockMutation: (
      input: useUpdateUserViewedVideoMutationVariables,
    ) => void = jest.fn();
    mockUseUpdateUserViewedVideo.mockReturnValue(mockMutation);

    setup(validVideoProps);

    // Wait enough time for the interval to fire so we can check the result
    act(() => {
      jest.advanceTimersByTime(2 * UPDATE_INTERVAL_S);
    });

    expect(mockMutation).toHaveBeenCalledWith({
      position: mockPosition,
    });
  });

  it('does not attempt to call the mutation if an invalid userID or videoID is provided', () => {
    // Set up a mock mutation function to be returned by the main mutation hook
    const mockMutation: (
      input: useUpdateUserViewedVideoMutationVariables,
    ) => void = jest.fn();
    mockUseUpdateUserViewedVideo.mockReturnValue(mockMutation);

    setup(invalidVideoProps);

    // Wait enough time for the interval to fire so we can check the result
    act(() => {
      jest.advanceTimersByTime(2 * UPDATE_INTERVAL_S);
    });

    expect(mockMutation).not.toHaveBeenCalled();
  });
});
