import type { ParsedUrlQuery } from 'querystring';
import type { LinkProps as NextLinkProps } from 'next/link';
import NextLink from 'next/link';
import type {
  AnchorHTMLAttributes,
  DetailedHTMLProps,
  FC,
  MouseEvent,
  PropsWithChildren,
} from 'react';
import type { InteractionDataPayload } from 'tachyon-event-tracker';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { convertGqlIdValuesToUnsafe } from 'tachyon-relay';
import type { CoreInteractiveProps } from 'twitch-core-ui';
import type { AsPathForRouteNameParams } from '../../linkPartsForRouteName';
import {
  isDynamicParams,
  linkPartsForRouteName,
} from '../../linkPartsForRouteName';
import { RouteName } from '../../routes';

type NextLinkPublicProps = Omit<NextLinkProps, 'as' | 'href'>;

type TachyonLinkPublicProps<T extends RouteName> = PropsWithChildren<
  AsPathForRouteNameParams<T> &
    NextLinkPublicProps &
    Partial<Omit<InteractionDataPayload, 'interactionType'>> & {
      query?: ParsedUrlQuery;
    }
>;

type TachyonLinkProps<T extends RouteName> = TachyonLinkPublicProps<T> & {
  anchorProps?: DetailedHTMLProps<
    AnchorHTMLAttributes<HTMLAnchorElement>,
    HTMLAnchorElement
  >;
};

/**
 * A wrapper around Next's Link component that provides type safe link generations.
 * All props from the original are available except "href" and "as" which are constructed
 * using "route" (in all cases) and "routeParams" (in the case of a dynamic route).
 *
 * Static Route:
 * <TachyonLink route={RouteName.Homepage}>Home</TachyonLink>
 *
 * Dynamic Route:
 * <TachyonLink route={RouteName.GameDirectory} routeParams={{ game: 'fortnite' }}>
 *   Fortnite
 * </TachyonLink>
 *
 * Query Params:
 * <TachyonLink route={RouteName.Homepage} query={{ tt_content: 'foo' }}>
 *   Home
 * </TachyonLink>
 */
export function TachyonLink<T extends RouteName>(
  props: TachyonLinkProps<T>,
): JSX.Element {
  const trackInteraction = useInteractionTracking();

  function onClick(e: MouseEvent<HTMLAnchorElement>): void {
    if (props.interactionContent) {
      trackInteraction({
        interaction: InteractionType.Click,
        interactionContent: props.interactionContent,
        interactionTargetPath: props.interactionTargetPath,
      });
    }

    props.anchorProps?.onClick?.(e);
  }

  // The use of "any" allows us to bypass the fact that TypeScript can't be sure
  // at this point as to what "params" requires as arguments due to type complexity.
  // However, Link's public prop API is type safe so we know for certain that if
  // routeParams are present, they are correct.
  const params: any = isDynamicParams(props)
    ? {
        route: props.route,
        // TODO: it is possible to have null params here, not sure why?
        // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
        routeParams:
          props.routeParams &&
          convertGqlIdValuesToUnsafe(props.routeParams as any),
      }
    : { route: props.route };

  // eslint-disable-next-line @typescript-eslint/no-unsafe-argument
  const { as, href } = linkPartsForRouteName(params, props.query);

  if (props.route === RouteName.External) {
    return <a {...props.anchorProps} href={as} onClick={onClick} />;
  }

  const linkProps: NextLinkPublicProps = {
    passHref: true,
    prefetch: props.prefetch,
    replace: props.replace,
    scroll: props.scroll,
    shallow: props.shallow,
  };

  return (
    <NextLink {...linkProps} as={as} href={href}>
      <a {...props.anchorProps} onClick={onClick} />
    </NextLink>
  );
}

TachyonLink.displayName = 'TachyonLink';

export function renderTachyonLink<T extends RouteName>(
  props: TachyonLinkPublicProps<T>,
): FC<CoreInteractiveProps> {
  const RenderTachyonLink: FC<CoreInteractiveProps> = ({
    // linkTo was purely to get Core UI to here so we discard it
    linkTo: _,
    ...coreInteractiveProps
  }) => <TachyonLink {...(props as any)} anchorProps={coreInteractiveProps} />;
  RenderTachyonLink.displayName = 'RenderTachyonLink';

  return RenderTachyonLink;
}
