import type { ParsedUrlQuery } from 'querystring';
import { stringify } from 'querystring';
import type { RouteLinkParams, RouteName } from '../routes';
import { RouteLinkBuilders } from '../routes';

/**
 * A conditional type helper for determining if a route is dynamic based on if its corresponding route builder takes
 * arguments.
 */
export type AsPathParams<T extends RouteName> = Parameters<
  typeof RouteLinkBuilders[T]
>[0] extends undefined
  ? never
  : Parameters<typeof RouteLinkBuilders[T]>[0];

export type StaticAsPathParams<T extends RouteName> = {
  route: T;
};

export type DynamicAsPathParams<T extends RouteName> = {
  route: T;
  routeParams: AsPathParams<T>;
};

/**
 * A conditional type helper for determining whether a Route requires static or dynamic params.
 */
export type AsPathForRouteNameParams<T extends RouteName> =
  AsPathParams<T> extends never
    ? StaticAsPathParams<T>
    : DynamicAsPathParams<T>;

type AsPathForRouteName = {
  /**
   * Corresponds to asPath.
   */
  as?: string;
  /**
   * Corresponds to pathname.
   */
  href: string;
};

/**
 * Helper function for creating link arguments for use with:
 *
 * <NextLink {...linkPartsForRouteName(...)}/>
 *
 * and
 *
 * const { href, as } = linkPartsForRouteName(...);
 * router.push(href, as);
 */
export function linkPartsForRouteName<T extends RouteName>(
  params: AsPathForRouteNameParams<T>,
  query?: ParsedUrlQuery,
): AsPathForRouteName {
  // The use of "any" allows us to bypass the fact that TypeScript can't be sure
  // at this point as to what "create" requires as arguments due to type complexity.
  // See: https://github.com/microsoft/TypeScript/issues/30581
  // However, Link's public prop API is type safe so we know for certain that if
  // routeParams are present, they are correct.
  // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
  const linkBuilder = RouteLinkBuilders[params.route] as any;

  // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
  const meta: RouteLinkParams = isDynamicParams(params)
    ? // eslint-disable-next-line @typescript-eslint/no-unsafe-call
      linkBuilder(params.routeParams)
    : // eslint-disable-next-line @typescript-eslint/no-unsafe-call
      linkBuilder();

  let href = meta.href;
  let as = meta.as;

  if (query) {
    const queryString = stringify(query);
    as = as ?? meta.href;
    as = `${as}?${queryString}`;
    href = `${href}?${queryString}`;
  }

  return {
    as,
    href,
  };
}

export function isDynamicParams<T extends RouteName>(
  params: DynamicAsPathParams<T> | StaticAsPathParams<T>,
): params is DynamicAsPathParams<T> {
  return 'routeParams' in params;
}
