import { random } from 'faker';
import { Enum } from 'tachyon-utils';
import { RouteLinkPathnames, RouteName } from '../routes';
import { routeNameFromRawPath } from '.';

describe(routeNameFromRawPath, () => {
  it.each(Enum.entries(RouteLinkPathnames))(
    'correctly matches for %s',
    (routeName: string, pathname: string) => {
      const rawPath = pathname.replace(/\[(\w+)\]/g, random.alphaNumeric(5));
      expect(routeNameFromRawPath(rawPath)).toEqual(routeName);
    },
  );

  type TestCases = {
    path: string;
    route: RouteName;
  };

  // sanity-checking the above generated tests and also validating
  // query param handling
  it.each`
    path                                     | route
    ${'/tfue'}                               | ${RouteName.Channel}
    ${'/tfue/home'}                          | ${RouteName.ChannelHome}
    ${'/directory'}                          | ${RouteName.GamesDirectory}
    ${'/directory/game/Fortnite'}            | ${RouteName.GameDirectory}
    ${'/search?term=counter'}                | ${RouteName.Search}
    ${'/app-shell?destination=%2Fdirectory'} | ${RouteName.AppShell}
  `('matches $path to $route', ({ path, route }: TestCases) => {
    expect(routeNameFromRawPath(path)).toEqual(route);
  });

  type NotFoundTestCase = {
    path: string;
  };

  it.each`
    path
    ${'/a/b/c/d'}
    ${'/a/b/c/d/e'}
    ${'/a/b/c/d/e/f'}
    ${'/directory/some/more/stuff'}
    ${'/clip/some/more/stuff'}
    ${'/videos/some/more/stuff'}
  `('matches $path to not found', ({ path }: NotFoundTestCase) => {
    expect(routeNameFromRawPath(path)).toEqual(RouteName.NotFound);
  });
});
