import { Enum } from 'tachyon-utils';

/**
 * A list of all route names in the app. This list must manually be updated when a new
 * page is added to the application. The values map to the route's equivalent Spade "location".
 */
export enum RouteName {
  AppShell = 'app_shell',
  Channel = 'channel',
  ChannelHome = 'channel_home',
  ChannelInterstitial = 'channel_interstitial',
  Clip = 'clip',
  ClipInterstitial = 'clip_interstitial',
  Error = 'error',
  External = 'external',
  GameDirectory = 'directory_game',
  GamesDirectory = 'directory_main_game',
  HealthCheck = 'health_check',
  Homepage = 'homepage',
  JoinConversation = 'join_conversation',
  Login = 'login',
  NotFound = 'not_found',
  Search = 'search',
  Settings = 'settings',
  SettingsChannel = 'settings_channel',
  SettingsChannelNotInterested = 'not_interested_channel',
  SettingsChannelReport = 'report_channel',
  SettingsDebugEnvironmentSelector = 'settings_debug_environment_selector',
  SettingsDebugExperimentManager = 'settings_debug_experiment_manager',
  SettingsDebugPathNavigator = 'settings_debug_path_navigator',
  SettingsEntityInformation = 'settings_entity_information',
  SettingsGameNotInterested = 'not_interested_game',
  SettingsHelp = 'settings_help',
  SettingsPrivacy = 'settings_privacy',
  SettingsProfile = 'settings_profile',
  SettingsTermsOfService = 'settings_terms_of_service',
  SettingsVod = 'settings_vod',
  SettingsVodReport = 'settings_vod_report',
  Vod = 'vod',
  VodInterstitial = 'vod_interstitial',
  WrongFormat = 'wrong_format',
}

type RouteMap = { [key in RouteName]: string };

/**
 * A mapping of Route names to their Next equivalent static "pathname". These
 * map directly to the folder structure of src/pages/*.
 */
export const RouteLinkPathnames: RouteMap = {
  [RouteName.AppShell]: '/app-shell',
  [RouteName.ChannelHome]: '/[login]/home',
  [RouteName.ChannelInterstitial]: '/[login]/interstitial',
  [RouteName.Clip]: '/[login]/clip/[slug]',
  [RouteName.ClipInterstitial]: '/[login]/clip/[slug]/interstitial',
  [RouteName.Error]: '/_error',
  [RouteName.External]: '/external',
  [RouteName.GameDirectory]: '/directory/game/[gameAlias]',
  [RouteName.GamesDirectory]: '/directory',
  [RouteName.HealthCheck]: '/_debug/running',
  [RouteName.Homepage]: '/',
  [RouteName.JoinConversation]: '/[login]/join-conversation',
  [RouteName.Login]: '/login',
  [RouteName.NotFound]: '/not-found',
  [RouteName.Search]: '/search',
  [RouteName.Settings]: '/settings',
  [RouteName.SettingsGameNotInterested]:
    '/settings/game/[gameAlias]/not-interested',
  [RouteName.SettingsChannel]: '/settings/channel/[login]',
  [RouteName.SettingsChannelNotInterested]:
    '/settings/channel/[login]/not-interested',
  [RouteName.SettingsChannelReport]: '/settings/channel/[login]/report',
  [RouteName.SettingsDebugEnvironmentSelector]:
    '/settings/debug/environment-selector',
  [RouteName.SettingsDebugExperimentManager]:
    '/settings/debug/experiment-manager',
  [RouteName.SettingsDebugPathNavigator]: '/settings/debug/path-navigator',
  [RouteName.SettingsEntityInformation]: '/settings/entity-information',
  [RouteName.SettingsHelp]: '/settings/help',
  [RouteName.SettingsPrivacy]: '/settings/privacy',
  [RouteName.SettingsProfile]: '/settings/profile',
  [RouteName.SettingsTermsOfService]: '/settings/terms-of-service',
  [RouteName.SettingsVod]: '/settings/videos/[videoId]',
  [RouteName.SettingsVodReport]: '/settings/videos/[videoId]/report',
  [RouteName.VodInterstitial]: '/videos/[videoId]/interstitial',
  [RouteName.Vod]: '/videos/[videoId]',
  [RouteName.WrongFormat]: '/wrong-format',
  // Channel must be last in this list to allow for proper matching behavior
  [RouteName.Channel]: '/[login]',
};

// Converts the Next-compatible paths to standard path-to-regexp-style paths
export const RouteLinkRawPaths = Enum.entries(RouteLinkPathnames).reduce<
  Partial<RouteMap>
>((acc, [route, pathname]) => {
  acc[route] = pathname.replace(/\[(\w+)\]/g, ':$1');
  return acc;
}, {}) as RouteMap;

export type RouteLinkParams = {
  as?: string;
  href: string;
};

/**
 * A mapping of route names to a matching link builder function. The arguments
 * for these functions also serve as the definition for which "routeParams" to
 * provide to the app's "Link" component.
 *
 * DO NOT USE DIRECTLY: Use `Link` ("src/components/framework/Link").
 *
 * Rules:
 * 1) All functions must return an object of { href: string; as?: string }.
 * 1) Static routes should take no arguments in their function and return only an "href" field.
 * 2) Dynamic routes should return both an "href" (path with placeholder) and "as" (path with dynamic value) field.
 */
export const RouteLinkBuilders = {
  [RouteName.AppShell]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.AppShell],
  }),
  [RouteName.ChannelHome]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}/home`,
    href: RouteLinkPathnames[RouteName.ChannelHome],
  }),
  [RouteName.ChannelInterstitial]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${login}/interstitial`,
    href: RouteLinkPathnames[RouteName.ChannelInterstitial],
  }),
  [RouteName.Clip]: ({
    login,
    slug,
  }: {
    login: string;
    slug: string;
  }): RouteLinkParams => ({
    as: `/${login}/clip/${slug}`,
    href: RouteLinkPathnames[RouteName.Clip],
  }),
  [RouteName.ClipInterstitial]: ({
    login,
    slug,
  }: {
    login: string;
    slug: string;
  }): RouteLinkParams => ({
    as: `/${login}/clip/${slug}/interstitial`,
    href: RouteLinkPathnames[RouteName.ClipInterstitial],
  }),
  [RouteName.Error]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Error],
  }),
  [RouteName.External]: ({ href }: { href: string }): RouteLinkParams => ({
    as: href,
    href,
  }),
  [RouteName.GameDirectory]: ({
    gameAlias,
  }: {
    gameAlias: string;
  }): RouteLinkParams => ({
    as: `/directory/game/${encodeURIComponent(gameAlias)}`,
    href: RouteLinkPathnames[RouteName.GameDirectory],
  }),
  [RouteName.GamesDirectory]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.GamesDirectory],
  }),
  [RouteName.HealthCheck]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.HealthCheck],
  }),
  [RouteName.Homepage]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Homepage],
  }),
  [RouteName.JoinConversation]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/${encodeURIComponent(login)}/join-conversation`,
    href: RouteLinkPathnames[RouteName.JoinConversation],
  }),
  [RouteName.Login]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Login],
  }),
  [RouteName.NotFound]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.NotFound],
  }),
  [RouteName.Search]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Search],
  }),
  [RouteName.Settings]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.Settings],
  }),
  [RouteName.SettingsGameNotInterested]: ({
    gameAlias,
  }: {
    gameAlias: string;
  }): RouteLinkParams => ({
    as: `/settings/game/${encodeURIComponent(gameAlias)}/not-interested`,
    href: RouteLinkPathnames[RouteName.SettingsGameNotInterested],
  }),
  [RouteName.SettingsChannel]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/settings/channel/${login}`,
    href: RouteLinkPathnames[RouteName.SettingsChannel],
  }),
  [RouteName.SettingsChannelReport]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/settings/channel/${login}/report`,
    href: RouteLinkPathnames[RouteName.SettingsChannelReport],
  }),
  [RouteName.SettingsChannelNotInterested]: ({
    login,
  }: {
    login: string;
  }): RouteLinkParams => ({
    as: `/settings/channel/${login}/not-interested`,
    href: RouteLinkPathnames[RouteName.SettingsChannelNotInterested],
  }),
  [RouteName.SettingsEntityInformation]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsEntityInformation],
  }),
  [RouteName.SettingsHelp]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsHelp],
  }),
  [RouteName.SettingsDebugEnvironmentSelector]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsDebugEnvironmentSelector],
  }),
  [RouteName.SettingsDebugExperimentManager]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsDebugExperimentManager],
  }),
  [RouteName.SettingsDebugPathNavigator]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsDebugPathNavigator],
  }),
  [RouteName.SettingsPrivacy]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsPrivacy],
  }),
  [RouteName.SettingsProfile]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsProfile],
  }),
  [RouteName.SettingsTermsOfService]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.SettingsTermsOfService],
  }),
  [RouteName.SettingsVod]: ({ id }: { id: string }): RouteLinkParams => ({
    as: `/settings/videos/${id}`,
    href: RouteLinkPathnames[RouteName.SettingsVod],
  }),
  [RouteName.SettingsVodReport]: ({ id }: { id: string }): RouteLinkParams => ({
    as: `/settings/videos/${id}/report`,
    href: RouteLinkPathnames[RouteName.SettingsVodReport],
  }),
  [RouteName.VodInterstitial]: ({ id }: { id: string }): RouteLinkParams => ({
    as: `/videos/${id}/interstitial`,
    href: RouteLinkPathnames[RouteName.VodInterstitial],
  }),
  [RouteName.Vod]: ({ id }: { id: string }): RouteLinkParams => ({
    as: `/videos/${id}`,
    href: RouteLinkPathnames[RouteName.Vod],
  }),
  [RouteName.WrongFormat]: (): RouteLinkParams => ({
    href: RouteLinkPathnames[RouteName.WrongFormat],
  }),
  [RouteName.Channel]: ({ login }: { login: string }): RouteLinkParams => ({
    as: `/${login}`,
    href: RouteLinkPathnames[RouteName.Channel],
  }),
};

/**
 * Converts a custom RouteName to a Next pathname.
 * Returns the NotFound pathname if there is no matching route.
 */
export function pathnameFromRouteName(name: RouteName): string {
  return RouteLinkPathnames[name] || RouteLinkPathnames[RouteName.NotFound];
}

/**
 * Converts a Next pathname to a RouteName. Returns NotFound if there is no matching path.
 */
export function routeNameFromPathname(pathname: string): RouteName {
  const routeName = Object.keys(RouteLinkPathnames).find(
    (route) => RouteLinkPathnames[route as RouteName] === pathname,
  ) as RouteName | undefined;

  return routeName ?? RouteName.NotFound;
}
