import { parse } from 'url';
import accepts from 'accepts';
import type { NextFunction, RequestHandler } from 'express';
import type { ExperimentBucket } from 'tachyon-experiments';
import { selectIntlData } from 'tachyon-intl-server';
import type {
  NextRequestHandler,
  TachyonRequest,
  TachyonRequestExtension,
  TachyonResponse,
} from 'tachyon-next-types';
import {
  createDeviceIDOnServer,
  generateExperimentBucketFromDeviceID,
  getDeviceIDOnServer,
  getExperimentBucket,
} from 'tachyon-server-utils';
import type { StarshotRequestExtensions } from '../../config';
import { EXPERIMENT_BUCKET_HEADER } from '../../config';

export const DEBUG_PATH_PREFIX = '/_debug';

export type ServerContext = Pick<
  TachyonRequestExtension<StarshotRequestExtensions>,
  'appEnvironment' | 'dynamicSettings' | 'platform'
>;

export function createAppRequestHandler(
  ctx: ServerContext,
  handler: NextRequestHandler,
): RequestHandler {
  return ((
    req: TachyonRequest<StarshotRequestExtensions>,
    res: TachyonResponse,
    _next: NextFunction,
  ): void => {
    if (process.env.NODE_ENV !== 'test') {
      console.log(`# Handling request for ${req.path}`);
    }

    const intlData = selectIntlData(accepts(req).languages());

    let deviceId: string | undefined;
    let experimentBucket: ExperimentBucket | undefined;

    // disregard debug paths since they aren't CDN-fronted and thus won't have user info
    if (!req.path.startsWith(DEBUG_PATH_PREFIX)) {
      // Ideally both deviceID and experimentBucket are always defined in production
      // since they should be managed by Fastly, and thus optimistically this
      // fallback logic only exists for local dev UX.
      deviceId = getDeviceIDOnServer(req, res) ?? createDeviceIDOnServer(res);
      experimentBucket =
        getExperimentBucket(req, {
          header: EXPERIMENT_BUCKET_HEADER,
          package: 'starshot',
        }) ?? generateExperimentBucketFromDeviceID(deviceId);
    }

    req.tachyon = {
      ...ctx,
      deviceId,
      experimentBucket,
      intlData,
    };

    handler(req, res, parse(req.url, true));
    // cast to narrow back down to express types
  }) as any as RequestHandler;
}
