import { resolve } from 'path';
import { URL } from 'url';
import cookieParser from 'cookie-parser';
import express from 'express';
import next from 'next';
import open from 'open';
import { fetchDynamicSettings } from 'tachyon-dynamic-settings';
import { Platform } from 'tachyon-environment';
import { prepareIntlDataCache } from 'tachyon-intl-server';
import {
  getTachyonEnvVar,
  setCookieDomainOnServer,
  tachyonServerMiddleware,
} from 'tachyon-server-utils';
import { Enum, getCurrentTwitchDomain } from 'tachyon-utils';
import type { StarshotDynamicSettings } from '../config';
import { DEFAULT_PLATFORM } from '../config';
import { RouteName, dynamicPathnames, pathnameFromRouteName } from '../routing';
import { createAppRequestHandler } from './appHandler';
import { isDevelopment } from './utils';

// polyfill WHATWG URL
// eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
(global as any).URL = URL;

const PORT = process.env.PORT ? parseInt(process.env.PORT, 10) : 3100;
const dev = isDevelopment();
// from `dist/server/index.js` to dir containing `.next/` & `messages/`
const appDir = dev ? undefined : resolve(__dirname, '../..');

// 15 minutes
const DYNAMIC_SETTINGS_TTL = 15 * 60 * 1000;

const appEnvironment = getTachyonEnvVar();
const platform =
  Enum.convertValueFromExternal(Platform, process.env.TACHYON_PLATFORM) ??
  DEFAULT_PLATFORM;

// istanbul ignore next: trivial + type-based enforcement
const fetchStarshotDynamicSettings =
  fetchDynamicSettings<StarshotDynamicSettings>({
    app: 'starshot',
    appEnvironment,
    appGroup: 'tachyon',
    // using the base settings for now
    processor: (settings) => settings,
  });

// top-level await unavailable until Node 14.8
// istanbul ignore next: trivial
prepareIntlDataCache({
  appDir,
  includePackages: ['tachyon-chat-ui', 'tachyon-more-ui'],
})
  .then(async () => {
    const dynamicSettings = await fetchStarshotDynamicSettings();
    const ctx = {
      appEnvironment,
      dynamicSettings,
      platform,
    };

    setInterval(() => {
      fetchStarshotDynamicSettings().then((settings) => {
        ctx.dynamicSettings = settings;
      });
    }, DYNAMIC_SETTINGS_TTL).unref();

    return ctx;
  })
  .then((ctx) => {
    const app = next({ dev, dir: appDir });
    const nextRequestHandler = app.getRequestHandler();
    const appRequestHandler = createAppRequestHandler(ctx, nextRequestHandler);

    app.prepare().then(() => {
      const server = express();
      server.use(
        tachyonServerMiddleware({
          badRequestMiddleware: {
            badRequestRedirectPath: pathnameFromRouteName(RouteName.NotFound),
            dynamicPathnames: dynamicPathnames(),
          },
        }),
      );
      server.use(cookieParser());

      // tell express to populate `req` convenience attributes from X-Forwarded-* headers
      // http://expressjs.com/en/4x/api.html#req.hostname
      server.enable('trust proxy');
      server.get('*', appRequestHandler);

      // allow setting alternate cookie domains (for staging environments only)
      const domain = getCurrentTwitchDomain(process.env.TACHYON_COOKIE_DOMAIN);
      if (domain && getTachyonEnvVar() === 'staging') {
        setCookieDomainOnServer(domain);
      }

      server.listen(PORT, () => {
        if (dev) {
          const devUrl = 'https://localhost.tv.twitch.tv';
          console.log(`\nListening at ${devUrl}\n`);
          open(devUrl, { app: { name: 'Chromium-38' } });
        }
      });
    });
  })
  .catch((error: Error) => {
    console.error(error.stack);
    process.exit(1);
  });
