import { parse, parseUrl } from 'query-string';
import { flattenHeaderOrParam } from 'tachyon-utils';
import {
  ACCESS_TOKEN_FRAGMENT_PARAM,
  REFRESH_TOKEN_FRAGMENT_PARAM,
} from '../../config';

type AccessAndRefreshToken = {
  accessToken?: string;
  refreshToken?: string;
};

/**
 * When passing auth tokens in with a URL it's important to use fragments
 * so that the auth token is not available server side. This is a helper
 * function for getting the auth and refresh token from the fragment.
 *
 * @param url the full url can be obtained from the router or window
 * @returns {(undefined|AccessAndRefreshToken)} undefined if the auth token is not present, else an object containing the auth token and the refresh token
 */
export function getAccessAndRefreshTokenFromUrlFragment(
  url: string,
): AccessAndRefreshToken | undefined {
  const { fragmentIdentifier } = parseUrl(url, {
    parseFragmentIdentifier: true,
  });

  if (!fragmentIdentifier) {
    return undefined;
  }

  const authData: AccessAndRefreshToken = {};
  const fragmentData = parse(fragmentIdentifier);

  if (ACCESS_TOKEN_FRAGMENT_PARAM in fragmentData) {
    authData.accessToken = flattenHeaderOrParam(
      fragmentData[ACCESS_TOKEN_FRAGMENT_PARAM],
    );
  }
  if (REFRESH_TOKEN_FRAGMENT_PARAM in fragmentData) {
    authData.refreshToken = flattenHeaderOrParam(
      fragmentData[REFRESH_TOKEN_FRAGMENT_PARAM],
    );
  }

  return authData.accessToken !== undefined ? authData : undefined;
}
