import { cy, expect, it } from 'local-cypress';
import type { JsonObject } from 'type-fest';

describe('GamesDirectory', () => {
  it('loads more content both on mount and on scroll (infinite scrolling)', () => {
    type CypressGqlRequest = {
      alias?: string;
      body: {
        query: string;
        variables: JsonObject;
      };
    };

    // Graphql url to which request is made
    const gqlURL = 'https://gql.twitch.tv/gql';

    // Aliases defined for refetch requests
    const refetchAlias = '@GameListRefetch';
    const allRefetchAliases = `${refetchAlias}.all`;

    // Function for creating aliases
    function routeHandler(req: CypressGqlRequest): void {
      if (req.body.query.includes('GameList_RefetchQuery')) {
        req.alias = refetchAlias.slice(1);
      }
    }
    // Spy on network request to gqlURL
    cy.intercept(gqlURL, routeHandler).as('gqlURL');

    // Used for tracking infinite scrolling state
    let previousListHeight = 0;

    // `any` type even though this is a known string; Cypress chain types don't
    // understand that certain matchers like have.css change the return type
    function haveHeightGreaterThanPrevious(heightPx: any): void {
      const newHeight = Number((heightPx as string).slice(0, -2));
      expect(newHeight).to.be.gt(previousListHeight);
      previousListHeight = newHeight;
    }

    function haveEquivalentHeight(heightPx: any): void {
      const newHeight = Number((heightPx as string).slice(0, -2));
      expect(newHeight).to.equal(previousListHeight);
    }

    cy.visit('/directory');

    // Ensure that GameList has a height on initial load (non-empty initial list)
    cy.get('main > div')
      .eq(1)
      .as('GameList')
      .should('have.css', 'height')
      .and(haveHeightGreaterThanPrevious);

    // Waiting for first request
    cy.wait(refetchAlias);

    // Ensure that GameList has grown after automatic refetch
    cy.get('@GameList')
      .should('have.css', 'height')
      .and(haveHeightGreaterThanPrevious);

    // Ensure that GameList does not grow after small scroll
    cy.get('main').scrollTo(0, 20);
    cy.get('@GameList').should('have.css', 'height').and(haveEquivalentHeight);

    // Ensure that second refetch does not fire without scrolling
    cy.get(allRefetchAliases).should('have.length', 1);
    cy.get('main').scrollTo('bottom');
    // Waiting for second request
    cy.wait(refetchAlias);

    // Ensure that GameList has grown after scroll-triggered refetch
    cy.get('@GameList')
      .should('have.css', 'height')
      .and(haveHeightGreaterThanPrevious);
  });
});
