// eslint-disable-next-line import/no-internal-modules
import type { CyHttpMessages } from 'cypress/types/net-stubbing';
import { Cypress, cy, expect, it } from 'local-cypress';
import { GQL_SANDBOX_ENDPOINT } from '../../src/config';

// type for array of services used for current page
type ServicesFailureData = {
  serviceDependencies: string[];
};

type CypressGqlRequest = Omit<CyHttpMessages.IncomingHttpRequest, 'body'> & {
  body: {
    query: string;
  };
};

type Service = string;
type Method = string;
type CypressGqlResponse = Omit<CyHttpMessages.IncomingHttpResponse, 'body'> & {
  body: {
    extensions: {
      queries: Array<{ method: `service.${Service}.${Method}` }>;
    };
  };
};

export interface CapabilityOptions {
  /**
   * Critical Capability test falls under
   */
  capability: string;
  /**
   * Specific feature we are testing
   */
  feature: string;
  /**
   * GraphQL query that represents the target capability being available
   */
  queryName: string;
  /**
   * Path from which to start testing for a capability
   */
  startPath: string;
}

function buildServicesFailureData(services: Set<string>): ServicesFailureData {
  const servicesObject: ServicesFailureData = {
    serviceDependencies: Array.from(services),
  };

  return servicesObject;
}

function getAppEntryPoint(startPath: string): string {
  return `/app-shell?destination=${encodeURIComponent(startPath)}`;
}
export function describeResilience(
  testSuiteName: string,
  callback: () => void,
  { capability, feature, queryName, startPath }: CapabilityOptions,
): void {
  const gqlProd = 'https://gql.twitch.tv/gql';
  // res flag set to 1 when retrieving services
  if (Cypress.env('resUp')) {
    describe(`Getting Services for ${testSuiteName}`, () => {
      const serviceDependencies: Set<string> = new Set();

      it('Getting Services', () => {
        cy.intercept(gqlProd, (req: CypressGqlRequest) => {
          req.url = GQL_SANDBOX_ENDPOINT;
          if (req.body.query.includes(queryName)) {
            req.headers['Twitch-Trace'] = 'extensions';
            req.alias = 'getServices';

            req.continue((res: CypressGqlResponse) => {
              res.body.extensions.queries.forEach(({ method }) => {
                // Parse service names and add to services set to avoid duplicates
                const [, service] = method.split('.');
                serviceDependencies.add(service);
              });
            });
          }
        });

        // we have to enter via the app-shell to guarantee all GQL requests
        // happen in the browser
        cy.visit(getAppEntryPoint(startPath));
        cy.wait('@getServices').then(() => {
          expect(serviceDependencies.size).to.be.gt(0);
          cy.writeFile(
            `cypress/fixtures/resilience/${testSuiteName}.json`,
            JSON.stringify(buildServicesFailureData(serviceDependencies)),
          );
        });

        // tests assertions against page without any failing services to ensure validity
        callback();
      });
    });
  }
  // res flag set to 0 when failing services
  else {
    const servicesArray =
      // eslint-disable-next-line @typescript-eslint/no-var-requires
      require(`../fixtures/resilience/${testSuiteName}.json`) as ServicesFailureData;

    describe(`${capability}--${feature}--${queryName}`, () => {
      const successScreenshotFileName = 'Success-Screenshot';
      it(`${successScreenshotFileName}`, () => {
        cy.visit(getAppEntryPoint(startPath));
        callback();
        cy.screenshot(successScreenshotFileName);
      });
      servicesArray.serviceDependencies.forEach((service) => {
        it(`Failing ${service}`, () => {
          // iterate through services failing them
          cy.intercept(gqlProd, (req: CypressGqlRequest) => {
            req.url = GQL_SANDBOX_ENDPOINT;
            if (req.body.query.includes(queryName)) {
              req.alias = 'failService';
              req.headers['Fail'] = service;
            }
          });

          // we have to enter via the app-shell to guarantee all GQL requests
          // happen in the browser
          cy.visit(getAppEntryPoint(startPath));
          cy.wait('@failService');

          // tests assertions against page for each failing service
          callback();
        });
      });
    });
  }
}

export {};
