/* eslint-disable import/order*/
const { join, resolve } = require('path');
const { readFileSync } = require('fs');
const { createHash } = require('crypto');
const { ProvidePlugin } = require('webpack');
const SentryWebpackPlugin = require('@sentry/webpack-plugin');
const withBundleAnalyzer = require('@next/bundle-analyzer')({
  enabled: !!process.env.ANALYZE,
});
const withWorkers = require('@zeit/next-workers');
const withFonts = require('next-fonts');
const withSourceMaps = require('@zeit/next-source-maps')({
  devtool: process.env.DOCKER_PRODUCTION ? 'hidden-source-map' : 'source-map',
});
const withPWA = require('next-pwa');
const es6Deps = require('tachyon-es6-deps-list');
const withTranspiledModules = require('next-transpile-modules')(es6Deps);
const CopyPlugin = require('copy-webpack-plugin');

const { buildId, nextAssetPrefix } = require('./env.config');

const { BGSYNC_QUERY_PARAM_STRING } =
  process.env.NODE_ENV === 'production'
    ? require('./dist/config')
    : require('./src/config');

/**
 * Create an md5 hash of the file contents
 */
const getRevision = (file) => {
  return createHash('md5').update(readFileSync(file)).digest('hex');
};

const APP_SHELL = '/app-shell';

const ENABLE_WORKBOX_LOGGING = false;

// These options are all fed into Workbox's GenerateSW plugin. Any of the
// options documented here will work:
// https://developers.google.com/web/tools/workbox/modules/workbox-webpack-plugin
//
// Default values for GenerateSW: https://github.com/GoogleChrome/workbox/blob/v6/packages/workbox-webpack-plugin/src/generate-sw.js
const WORKBOX_CONFIG = {
  // https://developers.google.com/web/tools/workbox/modules/workbox-navigation-preload
  // This is ignored on iOS and Safari. On Chrome, this will cause the initial request to bypass the service worker if it is still booting.
  navigationPreload: true,
  additionalManifestEntries: [
    // We use the buildId as the revision so the /app-shell page is cached each time there is a service worker update.
    // In dev, we churn the version so we don't get stuck with outdated changes
    {
      url: APP_SHELL,
      revision: buildId === 'tmw-dev' ? Date.now().toString() : buildId,
    },
    ...['/static/images/error-500.png', '/static/images/error-500@2x.png'].map(
      (url) => ({
        url,
        revision: getRevision(join(__dirname, 'public', url)),
      }),
    ),
  ],
  // exclude source map files from service worker installation
  buildExcludes: [/.map$/],
  /*
   * This is safe because the initial page is SSR'd. Once the new service worker has installed, we want it to immediately take control over the page so that subsequent navigations use the service worker.
   */
  skipWaiting: true,
  clientsClaim: true,
  inlineWorkboxRuntime: true,
  // mode = 'production' disables workbox logging, 'development' enables it.
  //
  // This otherwise defaults to the webpack setting, which is 'production' when running `yarn prodlike`.
  mode: ENABLE_WORKBOX_LOGGING ? 'development' : 'production',
  // https://github.com/GoogleChrome/workbox/issues/1407#issuecomment-463687081
  cleanupOutdatedCaches: true,
  // The entries in this list correspond to Workbox strategies:
  // https://developers.google.com/web/tools/workbox/reference-docs/latest/workbox.strategies#methods
  runtimeCaching: [
    {
      // The below is imposed by next-pwa
      // https://github.com/shadowwalker/next-pwa/issues/161
      urlPattern: '/this-is-not-used-but-next-pwa-requires-it',
      handler: 'NetworkOnly',
      options: {
        cacheName: 'start-url',
      },
    },
    {
      // RegExpRoute's will only handle cross-origin requests if they match the entire URL.
      urlPattern: new RegExp(`.*${BGSYNC_QUERY_PARAM_STRING}`),
      method: 'POST',
      handler: 'NetworkOnly',
      options: {
        backgroundSync: {
          name: 'default-sync',
          options: {
            // in minutes
            maxRetentionTime: 60 * 24,
          },
        },
      },
    },
    {
      // handle navigation fetches to allow navigationPreload to work
      // https://developers.google.com/web/tools/workbox/modules/workbox-routing
      urlPattern: (eventHolder) =>
        eventHolder.event.request.mode === 'navigate',
      handler: 'NetworkOnly',
      options: {
        precacheFallback: {
          fallbackURL: APP_SHELL,
        },
      },
    },
  ],
};

// Tomorrow next config
const NEXT_CONFIG = {
  webpack5: false,
  /**
   * Ensure that you validate changes by running `yarn prodlike` and inspecting
   * "public/service-worker.js".
   *
   * Docs: https://github.com/shadowwalker/next-pwa#readme
   */
  pwa: {
    dest: 'public',
    disable: process.env.NODE_ENV === 'development',
    // We take manual control of registration to ensure this process happens at
    // an optimal time and so that we can properly instrument it
    register: false,
    subdomainPrefix: nextAssetPrefix,
    sw: 'service-worker.esnext.js',
    ...WORKBOX_CONFIG,
  },
  // asset path prefixing
  // assetPrefix: process.env.NEXT_ASSET_PREFIX,
  // needs to be available at runtime in addition to build time.
  workerLoaderOptions: {
    filename: `static/[name]-[hash].js`,
    publicPath: `${nextAssetPrefix}_next/`,
  },

  // ignore typescript errors in local dev
  typescript: {
    ignoreDevErrors: true,
  },

  // set the build ID to the git hash for determinism
  generateBuildId: () => buildId,

  async headers() {
    return [
      {
        source: '/(.*)',
        headers: [
          {
            key: 'X-Frame-Options',
            value: 'deny',
          },
          {
            key: 'Content-Security-Policy',
            value: "frame-ancestors 'none'",
          },
        ],
      },
    ];
  },

  async redirects() {
    return [
      // https://github.com/vercel/next.js/issues/17189
      {
        source: '/_error',
        destination: '/not-found',
        permanent: false,
      },
    ];
  },

  // custom webpack config
  webpack(config, { defaultLoaders, dev, isServer }) {
    // Next 9.5 supports `next build --profile` but our current build process
    // is to complex to leverage that
    if (process.env.REACT_PROFILE) {
      alias['react-dom$'] = 'react-dom/profiling';
      alias['scheduler/tracing$'] = 'scheduler/tracing-profiling';
    }

    // set up aliases
    const alias = config.resolve.alias || {};

    config.resolve.alias = {
      ...alias,
      // react-router isn't used by Tachyon apps but is required by core-ui
      'react-router': 'tachyon-noop',
      'react-router-dom': 'tachyon-noop',
    };

    // ensure that bundle order doesn't result in a regeneratorRuntime error
    config.plugins.push(
      new ProvidePlugin({
        regeneratorRuntime: 'regenerator-runtime/runtime',
      }),
    );

    if (dev) {
      // See: https://git.xarth.tv/pages/emerging-platforms/tachyon/d/apps/tomorrow/processes/service-worker-development/
      config.plugins.push(
        new CopyPlugin({
          patterns: [
            {
              from: join(
                __dirname,
                'src',
                'workers',
                'development-service-worker.js',
              ),
              to: join(__dirname, 'public', 'service-worker.js'),
            },
          ],
        }),
      );
    }

    if (!isServer) {
      config.plugins.push(
        new SentryWebpackPlugin({
          dryRun: !process.env.SENTRY_AUTH_TOKEN,
          include: '.next',
          release: buildId,
          urlPrefix: '~/_next',
        }),
      );
    }

    return config;
  },
};

module.exports = (phase, { defaultConfig }) => {
  const baseConfig = withSourceMaps(
    withTranspiledModules(withPWA(withWorkers(withFonts(NEXT_CONFIG)))),
  );

  if (process.env.DOCKER_PRODUCTION) {
    return baseConfig;
  }

  return withBundleAnalyzer(baseConfig);
};
