# Latency Tracking and Optimization

In Tachyon Mobile Web, latency tracking is managed by the
tachyon-latency-tracking page.

## Optimizing Page Latency

### Deferred Rendering Of Below The Fold Content

Use `InfiniteList` in order to defer loading of components that are off screen.

```tsx
import { Component } from 'react';
import { InfiniteList } from 'components/common/InfiniteList';
import { AspectRatio, CardImage } from 'twitch-core-ui';
import { getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';

interface GamesListProps {
  games: Array<{ name: string; boxArtURL: string }>;
  loadMore: () => void;
}

/**
 * Rules for determining the size of your image based on a landscape or portrait
 * viewport.
 */
const BOX_ART_SIZES = [
  {
    mediaCondition: '(max-width: 540px) and (orientation: portrait)',
    size: '47vw',
  },
  { size: '24vw' },
];

class GamesList extends Component<GamesListProps> {
  public render(props: GamesListProps): JSX.Element {
    return (
      <InfiniteList
        itemRenderer={this.gameRenderer}
        length={this.props.games.length}
        loadMore={this.props.loadMore}
        pageSize={GAMES_PAGE_SIZE}
      />
    );
  }

  private gameRenderer = (idx: number): JSX.Element => {
    const game = this.props.games[idx];
    return (
      <div key={game.id}>
        <h3>{game.name}</h3>
        <CardImage
          {...getBoxArtSrcAndSrcSet(this.props.game.boxArtURL)}
          alt={game.name}
          aspect={AspectRatio.BoxArt}
          key={game.name}
          sizes={BOX_ART_SIZES}
        />
      </div>
    );
  };
}
```

## Instrumenting Page Latency

All pages in Tomorrow automatically have latency tracking enabled, and it is
controlled by the `TomorrowPage` boolean attribute `requiresJsForInteractivity`.
SSR pushes all pages into being interactive either when the HTML is done being
parsed or when the app JavaScript has booted. These definitions are ultimately
defined by the organization, but an easy way to test this assumption is loading
the page with JavaScript disabled and seeing if the Tier 1 experiences are
available yet.
