import type { PlayerError, PulsarCoreProps } from 'pulsar';
import { PulsarCore, usePlaybackState, usePlayerError } from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { isAndroidOS, useStaticEnvironment } from 'tachyon-environment';
import { useCustomLatency } from 'tachyon-latency-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { flattenHeaderOrParam, useConstCallback } from 'tachyon-utils';
import { Layout, Overflow, Position } from 'twitch-core-ui';
import {
  CLIENT_ID,
  PLAYER_DEBUG_MODE_QUERY_PARAM_KEY,
  PLAYER_DEBUG_MODE_QUERY_PARAM_VALUE,
} from '../../../config';
import { isWoodstock } from '../../../utils';
import { PlayerOverlayManager, spinnerOverlay } from '../overlays';
import type { Overlay } from '../overlays';

export type BasePlayerProps = Pick<PulsarCoreProps, 'loop' | 'startTime'> & {
  /**
   * A custom overlay to show over the player that blocks playback.
   */
  blockingOverlay: Overlay | null;
  onPlayerError: (error: PlayerError) => void;
  params: PulsarCoreProps['params'] | null;
};

/**
 * Manages the Pulsar Player backend, instrumentation + reporting, and content
 * agnostic playback UX.
 */
export const BasePlayer: FC<BasePlayerProps> = ({
  blockingOverlay,
  loop,
  onPlayerError,
  params,
  startTime,
}) => {
  const { client, common } = useStaticEnvironment();
  const reportLatencyEvent = useCustomLatency();
  const { currentPathname, currentQuery } = useRouterUtils();

  const playbackState = usePlaybackState();
  const playerError = usePlayerError();

  const reportPlayerReady = useConstCallback(reportLatencyEvent('playerReady'));

  useEffect(() => {
    if (playerError) {
      onPlayerError(playerError);
    }
  }, [playerError, onPlayerError]);

  useEffect(() => {
    if (playbackState === 'Playing') {
      reportLatencyEvent('playerPlaying')();
    }
  }, [playbackState, reportLatencyEvent]);

  const autoPlay = !blockingOverlay && !playerError;

  const playerDebugMode =
    flattenHeaderOrParam(currentQuery[PLAYER_DEBUG_MODE_QUERY_PARAM_KEY]) ===
    PLAYER_DEBUG_MODE_QUERY_PARAM_VALUE;

  // Block native controls for Woodstock on Android due to https://jira.twitch.com/browse/EMP-2951
  const supportsNativeControls = !(
    isWoodstock(currentQuery, currentPathname) && isAndroidOS(client.agentInfo)
  );

  // We don't want our full screen play button overlay to
  // have UI controls behind them that the user cannot interact with.
  const displayNativeControls =
    supportsNativeControls && playbackState !== 'Initializing' && autoPlay;

  const showSpinner =
    playbackState === 'Initializing' || playbackState === 'Buffering';
  const overlay = blockingOverlay ?? (showSpinner ? spinnerOverlay() : null);

  return (
    <Layout
      fullHeight
      fullWidth
      overflow={Overflow.Hidden}
      position={Position.Relative}
    >
      <PlayerOverlayManager overlay={overlay}>
        <PulsarCore
          autoPlay={autoPlay}
          controls={displayNativeControls}
          debugMode={playerDebugMode}
          environment={{
            benchmarkSessionID: client.sessionID,
            clientAPIID: CLIENT_ID,
            clientApp: common.clientApp,
            clientBuildID: common.appVersion,
            deviceID: client.deviceID,
            platform: common.platform,
          }}
          loop={loop}
          onFirstControlsAppear={reportPlayerReady}
          params={params}
          startTime={startTime}
        />
      </PlayerOverlayManager>
    </Layout>
  );
};

BasePlayer.displayName = 'BasePlayer';
