import type { ReactWrapper } from 'enzyme';
import type { PlaybackState, PulsarCoreParams } from 'pulsar';
import { PulsarCore, usePlaybackState, usePlayerError } from 'pulsar';
import type { Component } from 'react';
import {
  MockUserAgent,
  getAndroidEnvMock,
  getOtherEnvMock,
  getiOSEnvMock,
  useStaticEnvironment,
} from 'tachyon-environment';
import { mockRouterUtils, useRouterUtils } from 'tachyon-next-routing-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import {
  PLAYER_DEBUG_MODE_QUERY_PARAM_KEY,
  PLAYER_DEBUG_MODE_QUERY_PARAM_VALUE,
  WOODSTOCK_QUERY_PARAM_KEY,
  WOODSTOCK_QUERY_PARAM_VALUE,
} from '../../../config';
import { RouteLinkPathnames, RouteName } from '../../../routing';
import type { Overlay } from '../overlays';
import { PlayerOverlayManager, PlayerOverlayType } from '../overlays';
import type { BasePlayerProps } from '.';
import { BasePlayer } from '.';

const woodstockParams = {
  currentPathname: RouteLinkPathnames[RouteName.ChannelStandalone],
  currentQuery: {
    [WOODSTOCK_QUERY_PARAM_KEY]: WOODSTOCK_QUERY_PARAM_VALUE,
  },
};

function updatePlaybackState(
  wrapper: ReactWrapper<BasePlayerProps, {}, Component<{}, {}, any>>,
  state: PlaybackState,
) {
  mockUsePlaybackState.mockImplementation(() => state);
  wrapper.setProps({});
  wrapper.update();
}

function mockParams(): PulsarCoreParams {
  return {} as any;
}

function mockPlaybackBlocker(): Overlay {
  return {
    fatal: true,
    text: 'YOU SHALL NOT PASS',
    type: PlayerOverlayType.Message,
  };
}

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));

jest.mock('pulsar', () => ({
  ...jest.requireActual('pulsar'),
  PulsarCore: () => null,
  usePlaybackState: jest.fn(() => ({})),
  usePlayerError: jest.fn(),
}));

jest.mock('tachyon-next-routing-utils', () => ({
  ...jest.requireActual('tachyon-next-routing-utils'),
  useRouterUtils: jest.fn(() => mockRouterUtils),
}));

const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;
const mockUsePlaybackState = usePlaybackState as jest.Mock;
const mockUsePlayerError = usePlayerError as jest.Mock;
const mockUseRouterUtils = useRouterUtils as jest.Mock;

describe(BasePlayer, () => {
  const setup = createMountWrapperFactory(BasePlayer, () => ({
    blockingOverlay: null,
    onPlayerError: jest.fn(),
    params: mockParams(),
    playerId: 1,
  }));

  beforeEach(() => {
    mockUseRouterUtils.mockReturnValue(mockRouterUtils());
    mockUseStaticEnvironment.mockReturnValue(getOtherEnvMock());
    mockUsePlaybackState.mockReset();
    mockUsePlayerError.mockReset();
  });

  it('calls onPlayerError upon playerError', () => {
    const mockHandlePlayerError = jest.fn();
    mockUsePlayerError.mockImplementation(() => 'error');
    setup({ onPlayerError: mockHandlePlayerError });
    expect(mockHandlePlayerError).toHaveBeenCalledWith('error');
  });

  describe('overlay', () => {
    it('passes blockingOverlay to PlayerOverlayManager if blockingOverlay is not null', () => {
      const { props, wrapper } = setup({
        blockingOverlay: mockPlaybackBlocker(),
      });
      expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
        overlay: props.blockingOverlay,
      });
    });

    it('passes spinnerOverlay to PlayerOverlayManager if playbackState is Initializing', () => {
      mockUsePlaybackState.mockImplementationOnce(() => 'Initializing');
      const { wrapper } = setup();
      expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
        overlay: {
          type: PlayerOverlayType.Spinner,
        },
      });
    });

    it('passes spinnerOverlay to PlayerOverlayManager if playbackState is Buffering', () => {
      mockUsePlaybackState.mockImplementationOnce(() => 'Buffering');
      const { wrapper } = setup();
      expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
        overlay: {
          type: PlayerOverlayType.Spinner,
        },
      });
    });
  });

  describe('debug mode', () => {
    it('does not enable debug mode by default', () => {
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ debugMode: false });
    });

    it('does not enable debug mode for incorrect param value', () => {
      mockUseRouterUtils.mockReturnValue(
        mockRouterUtils({
          currentQuery: {
            [PLAYER_DEBUG_MODE_QUERY_PARAM_KEY]: 'therealderekt',
          },
        }),
      );
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ debugMode: false });
    });

    it('enables debug mode for when param value is true', () => {
      mockUseRouterUtils.mockReturnValue(
        mockRouterUtils({
          currentQuery: {
            [PLAYER_DEBUG_MODE_QUERY_PARAM_KEY]:
              PLAYER_DEBUG_MODE_QUERY_PARAM_VALUE,
          },
        }),
      );
      const { wrapper } = setup();
      expect(wrapper.find(PulsarCore)).toHaveProp({ debugMode: true });
    });
  });

  describe('playback', () => {
    it.each`
      type                | ua
      ${'iOS 12 (Phone)'} | ${MockUserAgent.iOS[12].iPhone}
      ${'iOS 12 (iPad)'}  | ${MockUserAgent.iOS[12].iPad}
      ${'iOS 10 (Phone)'} | ${MockUserAgent.iOS[10].iPhone}
      ${'iOS 10 (iPad)'}  | ${MockUserAgent.iOS[10].iPad}
      ${'Android 6'}      | ${MockUserAgent.Android[6].NexusTablet}
    `(
      'sets Pulsar props based on device autoplay capabilities for $type',
      ({ ua }: { type: string; ua: string }) => {
        mockUseStaticEnvironment.mockImplementationOnce(() =>
          getOtherEnvMock({ ua }),
        );
        const { wrapper } = setup();
        const pulsar = wrapper.find(PulsarCore);
        expect(pulsar).toHaveProp({
          autoPlay: true,
          controls: true,
        });
      },
    );

    describe('autoplay', () => {
      it('does not autoPlay while there are playback blockers', () => {
        const { wrapper } = setup({
          blockingOverlay: {
            fatal: false,
            text: 'PLAYBACK BLOCKED',
            type: PlayerOverlayType.Message,
          },
        });

        expect(wrapper.find(PulsarCore)).toHaveProp({
          autoPlay: false,
          controls: false,
        });

        wrapper.setProps({ blockingOverlay: null });
        wrapper.update();
        expect(wrapper.find(PulsarCore)).toHaveProp({
          autoPlay: true,
          controls: true,
        });
      });

      it('does not autoPlay when there is a player error', () => {
        mockUsePlayerError.mockImplementation(() => 'error');
        const { wrapper } = setup();
        expect(wrapper.find(PulsarCore)).toHaveProp({
          autoPlay: false,
        });
      });
    });

    // Failure cases are covered in the "tap to play" describe block
    describe('when playback succeeds', () => {
      it('hides the buffering overlay and displays native controls', () => {
        mockUsePlaybackState.mockImplementationOnce(() => 'Buffering');
        const { wrapper } = setup();
        updatePlaybackState(wrapper, 'Playing');

        expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
          overlay: null,
        });
        expect(wrapper.find(PulsarCore)).toHaveProp({ controls: true });
      });

      it('hides the buffering overlay and does not show native controls for Woodstock on Android', () => {
        mockUseStaticEnvironment.mockImplementation(() => ({
          ...getAndroidEnvMock(),
        }));
        mockUsePlaybackState.mockImplementationOnce(() => 'Buffering');
        mockUseRouterUtils.mockImplementation(() => ({
          ...mockRouterUtils,
          ...woodstockParams,
        }));
        const { wrapper } = setup();
        updatePlaybackState(wrapper, 'Playing');

        expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
          overlay: null,
        });
        expect(wrapper.find(PulsarCore)).toHaveProp({
          controls: false,
        });
      });

      it('hides the buffering overlay and does show native controls for Woodstock on iOS', () => {
        mockUseStaticEnvironment.mockImplementation(() => ({
          ...getiOSEnvMock(),
        }));
        mockUseRouterUtils.mockImplementation(() => ({
          ...mockRouterUtils,
          ...woodstockParams,
        }));
        const { wrapper } = setup();
        updatePlaybackState(wrapper, 'Playing');

        expect(wrapper.find(PlayerOverlayManager)).toHaveProp({
          overlay: null,
        });
        expect(wrapper.find(PulsarCore)).toHaveProp({
          controls: true,
        });
      });
    });
  });
});
