import { PlayerType, useClipAnalytics, useClipRestrictions } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useStaticEnvironment } from 'tachyon-environment';
import { clipPulsarParams } from 'tachyon-page-utils';
import { BasePlayer } from '../BasePlayer';
import { useDiscoveryTrackingEvent } from '../useDiscoveryTrackingEvent';
import type { ClipPlayer_clip } from './__generated__/ClipPlayer_clip.graphql';
import { useClipRestrictionOverlay } from './useClipRestrictionOverlay';

export type PulsarClipPlayerProps = {
  clip: ClipPlayer_clip;
};

export const PulsarClipPlayerBase: FC<PulsarClipPlayerProps> = ({ clip }) => {
  const { activeRestriction, handlePlayerError, removeRestriction } =
    useClipRestrictions(clip);
  const { overlay, playerId } = useClipRestrictionOverlay({
    removeRestriction,
    restriction: activeRestriction,
  });

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useDiscoveryTrackingEvent();

  useClipAnalytics({
    clip,
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  return (
    <BasePlayer
      blockingOverlay={overlay}
      key={playerId}
      loop
      onPlayerError={handlePlayerError}
      params={clipPulsarParams({
        clip,
      })}
    />
  );
};

PulsarClipPlayerBase.displayName = 'PulsarClipPlayer';

export const PulsarClipPlayer = createFragmentContainer(PulsarClipPlayerBase, {
  clip: graphql`
    fragment ClipPlayer_clip on Clip {
      id
      slug
      broadcaster {
        id
        login
        roles {
          isPartner
        }
        broadcastSettings {
          isMature
        }
        self {
          subscriptionBenefit {
            id
          }
        }
      }
      game {
        id
        name
      }
    }
  `,
});
