import type { Restriction } from 'pulsar';
import { VideoType, usePlayerController } from 'pulsar';
import { useIntl } from 'tachyon-intl';
import { logger } from 'tachyon-logger';
import { exhaustedCase, useForceUpdate } from 'tachyon-utils';
import type { Overlay } from '../../overlays';
import {
  matureContentOverlay,
  playbackErrorOverlay,
  regionalBlockOverlay,
  unknownRestrictionOverlay,
  usePlayerOverlay,
} from '../../overlays';

type UseClipRestrictionOverlayInput = {
  removeRestriction: (restriction: Restriction) => void;
  restriction: Restriction | null;
};

type UseClipRestrictionOverlay = {
  overlay: Overlay | null;
  playerId: number;
};

function logError(restriction: Restriction) {
  logger.error({
    category: 'getClipRestrictionOverlay',
    context: { contentType: VideoType.Clip },
    message: `Unexpected clip restriction '${restriction}'`,
    package: 'tomorrow',
  });
}

export function useClipRestrictionOverlay({
  removeRestriction,
  restriction,
}: UseClipRestrictionOverlayInput): UseClipRestrictionOverlay {
  const { formatMessage } = useIntl();
  const playerController = usePlayerController();
  const [playerId, incrementPlayerId] = useForceUpdate();
  let overlay = usePlayerOverlay();

  if (!restriction) {
    return { overlay, playerId };
  }

  switch (restriction) {
    case 'gql-mature': {
      overlay = matureContentOverlay({
        contentType: VideoType.Clip,
        formatMessage,
        onAccept: () => {
          removeRestriction('gql-mature');
          playerController?.play();
        },
      });
      break;
    }
    case 'playback-error': {
      overlay = playbackErrorOverlay(formatMessage, () => {
        removeRestriction('playback-error');
        incrementPlayerId();
      });
      break;
    }
    case 'player-auth-drm': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'player-auth-geo': {
      overlay = regionalBlockOverlay(formatMessage);
      break;
    }
    case 'player-auth-unknown': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'gql-sub-only':
    case 'gql-vod-muted-segments':
    case 'playback-ended':
    case 'player-auth-vod-sub-only': {
      logError(restriction);
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    default: {
      logError(restriction);
      overlay = exhaustedCase(
        restriction,
        unknownRestrictionOverlay(formatMessage),
      );
    }
  }

  return { overlay, playerId };
}
