import { PlayerType, useChannelRestrictions, useStreamAnalytics } from 'pulsar';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useStaticEnvironment } from 'tachyon-environment';
import { channelPulsarParams, getPlayingChannel } from 'tachyon-page-utils';
import { BasePlayer } from '../BasePlayer';
import { useDiscoveryTrackingEvent } from '../useDiscoveryTrackingEvent';
import type { StreamPlayer_channel } from './__generated__/StreamPlayer_channel.graphql';
import { useChannelRestrictionOverlay } from './useChannelRestrictionOverlay';

export type PulsarStreamPlayerProps = {
  channel: StreamPlayer_channel;
};

export const PulsarStreamPlayerBase: FC<PulsarStreamPlayerProps> = ({
  channel,
}) => {
  const { activeRestriction, handlePlayerError, removeRestriction } =
    useChannelRestrictions(channel);

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useDiscoveryTrackingEvent();
  useStreamAnalytics({
    channel,
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
  });

  const playingChannel = getPlayingChannel(channel);

  const { overlay, playerId } = useChannelRestrictionOverlay({
    channel,
    playingChannel,
    removeRestriction,
    restriction: activeRestriction,
  });

  return (
    <BasePlayer
      blockingOverlay={overlay}
      key={playerId}
      onPlayerError={handlePlayerError}
      params={channelPulsarParams(playingChannel)}
    />
  );
};

PulsarStreamPlayerBase.displayName = 'PulsarStreamPlayerBase';

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment StreamPlayer_channelBase on User {
    id
    displayName
    login
    offlineImageURL
    roles {
      isPartner
    }
    self {
      subscriptionBenefit {
        id
      }
    }
    stream {
      id
      game {
        id
        name
      }
      previewImageURL
      restrictionType
      self {
        canWatch
      }
      archiveVideo {
        id
      }
    }
    broadcastSettings {
      isMature
    }
  }
`;

// The $standalone Relay variable is to accommodate Woodstock. Relay errors
// when passing arguments to unmasked fragments which is why the videos query
// is not part of "StreamPlayer_channelBase"
export const PulsarStreamPlayer = createFragmentContainer(
  PulsarStreamPlayerBase,
  {
    channel: graphql`
      fragment StreamPlayer_channel on User
      @argumentDefinitions(
        standalone: { type: "Boolean", defaultValue: false }
      ) {
        ...StreamPlayer_channelBase @relay(mask: false)
        videos(sort: TIME, first: 3) @skip(if: $standalone) {
          edges {
            node {
              id
              previewThumbnailURL
              title
              status
            }
          }
        }
        hosting @skip(if: $standalone) {
          ...StreamPlayer_channelBase @relay(mask: false)
          videos(sort: TIME, first: 3) @skip(if: $standalone) {
            edges {
              node {
                id
                previewThumbnailURL
                title
                status
              }
            }
          }
        }
      }
    `,
  },
);

PulsarStreamPlayer.displayName = 'PulsarStreamPlayer';
