import type { Restriction } from 'pulsar';
import { VideoType, usePlayerController } from 'pulsar';
import { useIntl } from 'tachyon-intl';
import { logger } from 'tachyon-logger';
import type { PlayingChannel } from 'tachyon-page-utils';
import {
  exhaustedCase,
  getFormattedUserDisplayName,
  useForceUpdate,
} from 'tachyon-utils';
import type { Overlay } from '../../overlays';
import {
  channelOfflineOverlay,
  matureContentOverlay,
  nextVideoOverlay,
  playbackErrorOverlay,
  regionalBlockOverlay,
  subOnlyLiveOverlay,
  unknownRestrictionOverlay,
  usePlayerOverlay,
} from '../../overlays';
import { getNextVideo } from '../../utils';
import type { StreamPlayer_channel } from '../__generated__/StreamPlayer_channel.graphql';

type GetChannelRestrictionOverlayInput = {
  channel: StreamPlayer_channel;
  playingChannel: PlayingChannel<
    Omit<StreamPlayer_channel, ' $refType'>
  > | null;
  removeRestriction: (restriction: Restriction) => void;
  restriction: Restriction | null;
};

type UseChannelRestrictionOverlay = {
  overlay: Overlay | null;
  playerId: number;
};

function logError(restriction: Restriction) {
  logger.error({
    category: 'getChannelRestrictionOverlay',
    context: { contentType: VideoType.Stream },
    message: `Unexpected channel restriction '${restriction}'`,
    package: 'tomorrow',
  });
}

export function useChannelRestrictionOverlay({
  channel,
  playingChannel,
  removeRestriction,
  restriction,
}: GetChannelRestrictionOverlayInput): UseChannelRestrictionOverlay {
  const { formatMessage } = useIntl();
  const playerController = usePlayerController();
  const [playerId, incrementPlayerId] = useForceUpdate();
  let overlay = usePlayerOverlay();

  if (!playingChannel) {
    overlay = channelOfflineOverlay({
      formatMessage,
      // always use channel's offline image url because we aren't
      // cache-lagging here
      offlineImageURL: channel.offlineImageURL,
    });
    return { overlay, playerId };
  }

  if (!restriction) {
    return { overlay, playerId };
  }

  switch (restriction) {
    case 'gql-mature': {
      overlay = matureContentOverlay({
        contentType: VideoType.Stream,
        formatMessage,
        onAccept: () => {
          removeRestriction('gql-mature');
          playerController?.play();
        },
      });
      break;
    }
    case 'gql-sub-only': {
      overlay = subOnlyLiveOverlay(
        formatMessage,
        getFormattedUserDisplayName(playingChannel),
      );
      break;
    }
    case 'playback-ended':
      {
        const nextVideo =
          playingChannel.videos?.edges &&
          getNextVideo({
            omitVideoID: playingChannel.stream.archiveVideo?.id ?? null,
            recentVideos: playingChannel.videos.edges,
          });

        if (nextVideo) {
          overlay = nextVideoOverlay({
            creatorName: playingChannel.displayName,
            formatMessage,
            onTapOverlay: () => {
              removeRestriction('playback-ended');
            },
            video: nextVideo,
          });
        }
        overlay = channelOfflineOverlay({
          formatMessage,
          offlineImageURL: playingChannel.offlineImageURL,
        });
      }
      break;
    case 'playback-error': {
      overlay = playbackErrorOverlay(formatMessage, () => {
        removeRestriction('playback-error');
        incrementPlayerId();
      });
      break;
    }
    case 'player-auth-drm': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'player-auth-geo': {
      overlay = regionalBlockOverlay(formatMessage);
      break;
    }
    case 'player-auth-unknown': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'gql-vod-muted-segments':
    case 'player-auth-vod-sub-only': {
      logError(restriction);
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    default: {
      logger.error(restriction);
      overlay = exhaustedCase(
        restriction,
        unknownRestrictionOverlay(formatMessage),
      );
    }
  }

  return { overlay, playerId };
}
