import {
  PlayerType,
  usePlaybackState,
  useVodAnalytics,
  useVodRestrictions,
} from 'pulsar';
import type { FC } from 'react';
import { useEffect } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useStaticEnvironment } from 'tachyon-environment';
import { useStartTime, vodPulsarParams } from 'tachyon-page-utils';
import { PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY } from '../../../config';
import { BasePlayer } from '../BasePlayer';
import { useDiscoveryTrackingEvent } from '../useDiscoveryTrackingEvent';
import type { VodPlayer_video } from './__generated__/VodPlayer_video.graphql';
import { useVodRestrictionOverlay } from './useVodRestrictionOverlay';

export type {
  VideoStatus,
  VodPlayer_video,
} from './__generated__/VodPlayer_video.graphql';

export type PulsarVodPlayerProps = {
  video: VodPlayer_video;
};

export const PulsarVodPlayerBase: FC<PulsarVodPlayerProps> = ({ video }) => {
  const {
    activeRestriction,
    addRestriction,
    handlePlayerError,
    removeRestriction,
  } = useVodRestrictions(video);
  const startTime = useStartTime(PLAYER_INITIAL_OFFSET_QUERY_PARAM_KEY);
  const playbackState = usePlaybackState();
  const { overlay, playerId } = useVodRestrictionOverlay({
    removeRestriction,
    restriction: activeRestriction,
    video,
  });

  const {
    client: { sessionID },
  } = useStaticEnvironment();

  const onTrackingEvent = useDiscoveryTrackingEvent();
  useVodAnalytics({
    currentUser: null,
    onTrackingEvent,
    tracking: {
      playerType: PlayerType.Pulsar,
      sessionId: sessionID,
    },
    video,
  });

  useEffect(() => {
    if (playbackState === 'Ended') {
      addRestriction('playback-ended');
    }
  }, [addRestriction, playbackState]);

  return (
    <BasePlayer
      blockingOverlay={overlay}
      key={playerId}
      onPlayerError={handlePlayerError}
      params={vodPulsarParams({
        vod: video,
      })}
      startTime={startTime}
    />
  );
};

PulsarVodPlayerBase.displayName = 'PulsarVodPlayerBase';

// We set videos object "first" arg value to 8, since the VodPanel requires that many and the
// videos object arguments need to match
export const PulsarVodPlayer = createFragmentContainer(PulsarVodPlayerBase, {
  video: graphql`
    fragment VodPlayer_video on Video {
      id
      broadcastType
      owner {
        id
        displayName
        login
        roles {
          isPartner
        }
        broadcastSettings {
          isMature
        }
        self {
          subscriptionBenefit {
            id
          }
        }
        subscriptionProducts {
          displayName
          hasSubonlyVideoArchive
        }
        videos(sort: TIME, first: 8) {
          edges {
            node {
              id
              previewThumbnailURL
              title
              status
            }
          }
        }
      }
      game {
        id
        name
      }
      muteInfo {
        mutedSegmentConnection {
          nodes {
            duration
          }
        }
      }
      resourceRestriction {
        type
      }
    }
  `,
});
