import type { Restriction } from 'pulsar';
import { VideoType, usePlayerController } from 'pulsar';
import { useIntl } from 'tachyon-intl';
import { logger } from 'tachyon-logger';
import { exhaustedCase, useForceUpdate } from 'tachyon-utils';
import type { Overlay } from '../../overlays';
import {
  matureContentOverlay,
  mutedSegmentsWarning,
  nextVideoOverlay,
  playbackErrorOverlay,
  regionalBlockOverlay,
  subOnlyVideoOverlay,
  unknownRestrictionOverlay,
  usePlayerOverlay,
} from '../../overlays';
import { getNextVideo } from '../../utils';
import type { VodPlayer_video } from '../__generated__/VodPlayer_video.graphql';

type UseVodRestrictionInput = {
  removeRestriction: (restriction: Restriction) => void;
  restriction: Restriction | null;
  video: VodPlayer_video;
};

type UseVodRestrictionOverlay = {
  overlay: Overlay | null;
  playerId: number;
};

export function useVodRestrictionOverlay({
  removeRestriction,
  restriction,
  video,
}: UseVodRestrictionInput): UseVodRestrictionOverlay {
  const { formatMessage } = useIntl();
  const playerController = usePlayerController();
  const [playerId, incrementPlayerId] = useForceUpdate();
  let overlay = usePlayerOverlay();

  if (!restriction) {
    return { overlay, playerId };
  }

  switch (restriction) {
    case 'gql-mature': {
      overlay = matureContentOverlay({
        contentType: VideoType.Video,
        formatMessage,
        onAccept: () => {
          removeRestriction('gql-mature');
          playerController?.play();
        },
      });
      break;
    }
    case 'gql-sub-only': {
      overlay = subOnlyVideoOverlay(formatMessage);
      break;
    }
    case 'gql-vod-muted-segments': {
      overlay = mutedSegmentsWarning(formatMessage, () => {
        playerController?.setMuted(false);
        removeRestriction('gql-vod-muted-segments');
        playerController?.play();
      });
      break;
    }
    case 'playback-ended': {
      if (video.owner) {
        const nextVideo =
          video.owner.videos?.edges &&
          getNextVideo({
            omitVideoID: video.id,
            recentVideos: video.owner.videos.edges,
          });

        if (nextVideo) {
          overlay = nextVideoOverlay({
            creatorName: video.owner.displayName,
            formatMessage,
            onTapOverlay: () => {
              removeRestriction('playback-ended');
            },
            video: nextVideo,
          });
        }
      }
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'playback-error': {
      overlay = playbackErrorOverlay(formatMessage, () => {
        removeRestriction('playback-error');
        incrementPlayerId();
      });
      break;
    }
    case 'player-auth-drm': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'player-auth-geo': {
      overlay = regionalBlockOverlay(formatMessage);
      break;
    }
    case 'player-auth-unknown': {
      overlay = unknownRestrictionOverlay(formatMessage);
      break;
    }
    case 'player-auth-vod-sub-only': {
      overlay = subOnlyVideoOverlay(formatMessage);
      break;
    }
    default: {
      logger.error({
        category: 'getVodRestrictionOverlay',
        context: { contentType: VideoType.Video },
        // eslint-disable-next-line @typescript-eslint/restrict-template-expressions
        message: `Unexpected vod restriction '${restriction}'`,
        package: 'tomorrow',
      });
      overlay = exhaustedCase(
        restriction,
        unknownRestrictionOverlay(formatMessage),
      );
    }
  }

  return { overlay, playerId };
}
