import type { FC } from 'react';
import { memo } from 'react';
import type { SrcAndSrcSet } from 'tachyon-more-ui';
import type { CoreButtonProps, CoreImageSize } from 'twitch-core-ui';
import {
  AlignContent,
  AlignItems,
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreImage,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Position,
  TextAlign,
} from 'twitch-core-ui';
import { ContrastLayout } from '../ContrastLayout';
import { PlayerOverlay } from '../PlayerOverlay';

interface ButtonOpts
  extends Partial<Pick<CoreButtonProps, 'onClick' | 'renderLink'>> {
  text: string;
}

interface BackgroundImage {
  sizes: CoreImageSize[];
  srcAndSrcSet: SrcAndSrcSet;
}

export interface MessageOverlayProps {
  /**
   * An image to render behind the overlay text / button. As a side-effect,
   * overlay text will be placed in the top right corner and rendered with a
   * background for contrast.
   */
  backgroundImage?: BackgroundImage;
  /**
   * A call to action that allows a user to take remediation steps for the
   * overlay message. Exposes a subset of CoreButton props.
   */
  button?: ButtonOpts;
  text: string;
}

export const MessageOverlayBase: FC<MessageOverlayProps> = ({
  backgroundImage,
  button,
  text,
}) => {
  const btn = button ? (
    <Layout margin={{ top: 2 }}>
      <Button
        linkTo={button.renderLink && '/deferToRenderLink'}
        onClick={button.onClick}
        renderLink={button.renderLink}
        size={ButtonSize.Large}
        variant={ButtonType.Primary}
      >
        {button.text}
      </Button>
    </Layout>
  ) : null;

  let overlayText = (
    <CoreText color={Color.Overlay} fontSize={FontSize.Size5}>
      {text}
    </CoreText>
  );

  if (backgroundImage) {
    overlayText = (
      <ContrastLayout attachRight attachTop position={Position.Absolute}>
        {overlayText}
      </ContrastLayout>
    );
  }

  // 1) Since this is a background image, position it absolutely so it does not
  //    push the other overlay content off screen
  // 2) The image "alt" attribute is purposely '' to tell screen-readers, etc
  //    that it is non-content
  const bgImage = backgroundImage?.srcAndSrcSet.src && (
    <Layout
      alignContent={AlignContent.Center}
      alignItems={AlignItems.Center}
      display={Display.Flex}
      fullHeight
      fullWidth
      position={Position.Absolute}
    >
      <CoreImage
        alt={''}
        sizes={backgroundImage.sizes}
        {...backgroundImage.srcAndSrcSet}
      />
    </Layout>
  );

  return (
    <PlayerOverlay>
      {bgImage}
      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullHeight
        fullWidth
        justifyContent={JustifyContent.Center}
        padding={{ x: 1 }}
        textAlign={TextAlign.Center}
      >
        {overlayText}
        {btn}
      </Layout>
    </PlayerOverlay>
  );
};
MessageOverlayBase.displayName = 'MessageOverlayBase';

export const MessageOverlay = memo(MessageOverlayBase);
