import { internet, lorem } from 'faker';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Button, CoreImage, CoreText } from 'twitch-core-ui';
import { PlayerOverlay } from '../PlayerOverlay';
import { MessageOverlayBase } from '.';

describe(MessageOverlayBase, () => {
  const setup = createShallowWrapperFactory(MessageOverlayBase, () => ({
    text: lorem.words(),
  }));

  it('does not allow tap through', () => {
    const { wrapper } = setup();
    expect(wrapper.find(PlayerOverlay).props().tapThrough).toBeUndefined();
  });

  it('shows text only', () => {
    const { props, wrapper } = setup();

    expect(wrapper.find(CoreText).childAt(0).text()).toEqual(props.text);
    expect(wrapper.find(Button)).not.toExist();
  });

  it('shows text and button', () => {
    const { props, wrapper } = setup({
      button: {
        onClick: jest.fn(),
        renderLink: () => lorem.word(),
        text: lorem.words(),
      },
    });

    expect(wrapper.find(CoreText).childAt(0).text()).toEqual(props.text);
    const buttonWrapper = wrapper.find(Button);
    expect(buttonWrapper.childAt(0).text()).toEqual(props.button!.text);
    expect(buttonWrapper).toHaveProp({
      onClick: props.button!.onClick,
      renderLink: props.button!.renderLink,
    });
  });

  it('renders a background image when one is specified and pushes overlay text to the top right', () => {
    const expectedSizes = [{}];
    const { props, wrapper } = setup({
      backgroundImage: {
        sizes: expectedSizes,
        srcAndSrcSet: {
          src: internet.url(),
          srcSet: {
            '1x': internet.url(),
          },
        },
      },
    });

    expect(wrapper.find(CoreImage)).toHaveProp({
      sizes: expectedSizes,
      src: props.backgroundImage!.srcAndSrcSet.src,
      srcSet: props.backgroundImage!.srcAndSrcSet.srcSet,
    });
    expect(wrapper.find(CoreText).parent()).toHaveProp({
      attachRight: true,
      attachTop: true,
    });
  });

  it('does not show a background image when one is passed as empty string (non-nullable typing is a lie!)', () => {
    const { wrapper } = setup({
      backgroundImage: {
        sizes: [],
        srcAndSrcSet: { src: '' },
      },
    });

    expect(wrapper.find(CoreImage)).not.toExist();
  });
});
