import type { FC } from 'react';
import { memo } from 'react';
import styled from 'styled-components';
import { getOverlaySrcAndSrcSet } from 'tachyon-more-ui';
import {
  AlignItems,
  Background,
  Button,
  ButtonType,
  Color,
  Column,
  CoreImage,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Grid,
  InjectLayout,
  JustifyContent,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { PlayerOverlay } from '../PlayerOverlay';

export const ScNextVideoPreview = styled(CoreImage)`
  max-height: 80%;
`;

export interface NextVideoOverlayVideo {
  altText: string;
  displayText: string;
  id: string;
  /**
   * Resizable URL template containing "{width}" and "{height}" placeholders.
   */
  previewImageUrlTemplate: string;
  title: string;
}

export interface NextVideoOverlayProps {
  dismissText: string;
  onTapOverlay: () => void;
  video: NextVideoOverlayVideo;
}

/**
 * Overlay that presents the user with a new video to watch.
 */
export const NextVideoOverlayBase: FC<NextVideoOverlayProps> = ({
  dismissText,
  onTapOverlay,
  video,
}) => {
  const srcAndSet = getOverlaySrcAndSrcSet(video.previewImageUrlTemplate);

  return (
    <PlayerOverlay onTapOverlay={onTapOverlay}>
      <Layout
        alignItems={AlignItems.Center}
        background={Background.Overlay}
        display={Display.Flex}
        fullHeight
        fullWidth
        justifyContent={JustifyContent.Center}
      >
        <Grid
          alignItems={AlignItems.Center}
          justifyContent={JustifyContent.Center}
        >
          <Column cols={{ default: 8, xs: 6 }}>
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              ellipsis
              flexDirection={FlexDirection.Row}
              justifyContent={JustifyContent.Between}
              overflow={Overflow.Hidden}
              padding={{ y: 0.5 }}
            >
              <CoreText
                color={Color.Overlay}
                ellipsis
                fontSize={FontSize.Size3}
              >
                {video.displayText}
              </CoreText>
              <Button overlay variant={ButtonType.Primary}>
                {dismissText}
              </Button>
            </Layout>
            <InjectLayout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              flexShrink={4}
              justifyContent={JustifyContent.Center}
            >
              <CoreLink
                linkTo="/deferToRenderLink"
                renderLink={renderTachyonLink({
                  interactionContent: 'next-video-overlay-link',
                  interactionTargetPath: `/videos/${video.id}`,
                  route: RouteName.Vod,
                  routeParams: { videoID: video.id },
                })}
              >
                <ScNextVideoPreview {...srcAndSet} alt={video.altText} />
              </CoreLink>
            </InjectLayout>
            <Layout ellipsis overflow={Overflow.Hidden}>
              <CoreText
                color={Color.Overlay}
                ellipsis
                fontSize={FontSize.Size4}
              >
                {video.title}
              </CoreText>
            </Layout>
          </Column>
        </Grid>
      </Layout>
    </PlayerOverlay>
  );
};
NextVideoOverlayBase.displayName = 'NextVideoOverlayBase';

export const NextVideoOverlay = memo(NextVideoOverlayBase);
