import type { FC } from 'react';
import { Display, Layout, Position } from 'twitch-core-ui';
import type { FullScreenControlsOverlayProps } from '../FullScreenControlsOverlay';
import { FullScreenControlsOverlay } from '../FullScreenControlsOverlay';
import type { MessageOverlayProps } from '../MessageOverlay';
import { MessageOverlay } from '../MessageOverlay';
import type { NextVideoOverlayProps } from '../NextVideoOverlay';
import { NextVideoOverlay } from '../NextVideoOverlay';
import type { PlayerControlsOverlayProps } from '../PlayerControlsOverlay';
import { PlayerControlsOverlay } from '../PlayerControlsOverlay';
import { SpinnerOverlay } from '../SpinnerOverlay';

export enum PlayerOverlayType {
  Spinner,
  Message,
  PlayerControls,
  NextVideo,
  FullScreenPlayerControls,
}

/**
 * Causes a spinner overlay to appear with a transparent background.
 */
export interface OverlaySpinner {
  type: PlayerOverlayType.Spinner;
}

/**
 * An overlay for displaying a text message with an optional CTA buttons and
 * various styling options like setting a background image.
 */
export interface OverlayMessage extends MessageOverlayProps {
  /**
   * Fatal overlays cannot be dismissed and will not result in playback. As a
   * a result they will result in the child components either being unmounted or
   * never mounted in the first place (depending on when the overlay appears in
   * the component lifecycle).
   */
  fatal: boolean;
  type: PlayerOverlayType.Message;
}

/**
 * An overlay that appears over a loaded video player.
 */
export interface OverlayPlayerControls extends PlayerControlsOverlayProps {
  type: PlayerOverlayType.PlayerControls;
}

/**
 * Causes an overlay that should be used to link to another video
 */
export interface OverlayNextVideo extends NextVideoOverlayProps {
  type: PlayerOverlayType.NextVideo;
}

/**
 * Causes an overlay that should be used to control player state with a full
 * screen icon
 */
export interface OverlayFullScreenControls
  extends FullScreenControlsOverlayProps {
  type: PlayerOverlayType.FullScreenPlayerControls;
}

export type Overlay =
  | OverlayFullScreenControls
  | OverlayMessage
  | OverlayNextVideo
  | OverlayPlayerControls
  | OverlaySpinner;

export interface PlayerOverlayManagerProps {
  overlay: Overlay | null;
}

/**
 * A component meant to wrap a player so that it can be hidden when a blocking
 * message (such as age-gating) needs to be displayed. It uses `display: none`
 * to keep the player elements in the DOM but not visible to prevent unmount/
 * mount cycles. Visibility is controlled by whether a non-null overlay prop is
 * passed.
 */
export const PlayerOverlayManager: FC<PlayerOverlayManagerProps> = ({
  children: player,
  overlay,
}) => {
  let playerOverlay: JSX.Element | null;
  let playerDisplay: Display | undefined;

  if (!overlay) {
    playerOverlay = null;
  } else if (overlay.type === PlayerOverlayType.Spinner) {
    playerOverlay = <SpinnerOverlay />;
  } else if (overlay.type === PlayerOverlayType.PlayerControls) {
    playerOverlay = <PlayerControlsOverlay {...overlay} />;
  } else if (overlay.type === PlayerOverlayType.FullScreenPlayerControls) {
    playerOverlay = <FullScreenControlsOverlay {...overlay} />;
  } else if (overlay.type === PlayerOverlayType.NextVideo) {
    playerOverlay = <NextVideoOverlay {...overlay} />;
  } else {
    playerOverlay = <MessageOverlay {...overlay} />;
    playerDisplay = Display.Hide;

    // If fatal, we should clean up the player since we want to avoid any
    // unnecessary tracking events, state, etc
    if (overlay.fatal) {
      return playerOverlay;
    }
  }

  return (
    <>
      <Layout
        display={playerDisplay}
        fullHeight
        fullWidth
        position={Position.Absolute}
      >
        {player}
      </Layout>
      {playerOverlay}
    </>
  );
};

PlayerOverlayManager.displayName = 'PlayerMessageOverlay';
