import { datatype, lorem } from 'faker';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { Display, SVGAsset } from 'twitch-core-ui';
import { FullScreenControlsOverlay } from '../FullScreenControlsOverlay';
import { NextVideoOverlay } from '../NextVideoOverlay';
import { AssetLocation, PlayerControlsOverlay } from '../PlayerControlsOverlay';
import { SpinnerOverlay } from '../SpinnerOverlay';
import { PlayerOverlayManager, PlayerOverlayType } from '.';

describe(PlayerOverlayManager, () => {
  describe('with a null overlay prop', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: null,
    }));

    it('shows children with default display settings', () => {
      const { wrapper } = setup();

      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: undefined,
      });
    });
  });

  describe('with a spinner overlay', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: { type: PlayerOverlayType.Spinner },
    }));

    it('renders the spinner and children with default display settings', () => {
      const { wrapper } = setup();
      expect(wrapper.find(SpinnerOverlay)).toExist();
      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: undefined,
      });
    });
  });

  describe('with a full screen controls overlay', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: {
        asset: SVGAsset.Muted,
        onTapOverlay: () => undefined,
        text: 'Some text',
        type: PlayerOverlayType.FullScreenPlayerControls,
      },
    }));

    it('renders the controls and children with default display settings', () => {
      const { props, wrapper } = setup();
      expect(wrapper.find(FullScreenControlsOverlay).props()).toMatchObject(
        props.overlay!,
      );
      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: undefined,
      });
    });
  });

  describe('with a player controls overlay', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: {
        asset: SVGAsset.Muted,
        assetLocation: AssetLocation.Left,
        onTapOverlay: () => undefined,
        text: 'Some text',
        type: PlayerOverlayType.PlayerControls,
      },
    }));

    it('renders the controls and children with default display settings', () => {
      const { props, wrapper } = setup();
      expect(wrapper.find(PlayerControlsOverlay).props()).toMatchObject(
        props.overlay!,
      );
      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: undefined,
      });
    });
  });

  describe('with a message overlay', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: {
        fatal: true,
        text: lorem.words(),
        type: PlayerOverlayType.Message,
      },
    }));

    it('renders the message overlay directly for fatal messages', () => {
      const { props, wrapper } = setup({ overlay: { fatal: true } });
      expect(wrapper.props()).toMatchObject(props.overlay!);
      expect(wrapper.find('#some-child')).not.toExist();
    });

    it('renders the children as hidden for non-fatal messages', () => {
      const { wrapper } = setup({ overlay: { fatal: false } });

      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: Display.Hide,
      });
    });
  });

  describe('with a next video overlay', () => {
    const setup = createShallowWrapperFactory(PlayerOverlayManager, () => ({
      children: <div id="some-child" />,
      overlay: {
        dismissText: 'Dismiss',
        onTapOverlay: jest.fn(),
        type: PlayerOverlayType.NextVideo,
        video: {
          altText: lorem.words(),
          displayText: lorem.words(),
          id: datatype.uuid(),
          previewImageUrlTemplate: lorem.word(),
          title: lorem.sentence(),
        },
      },
    }));

    it('renders the overlay with default display settings', () => {
      const { wrapper } = setup();
      expect(wrapper.find(NextVideoOverlay)).toExist();
      expect(wrapper.find('#some-child').parent()).toHaveProp({
        display: undefined,
      });
    });
  });
});
