import type { FC } from 'react';
import { memo } from 'react';
import styled from 'styled-components';
import type { SVGAsset } from 'twitch-core-ui';
import {
  AlignItems,
  AlignSelf,
  Color,
  CoreText,
  Display,
  FlexDirection,
  JustifyContent,
  Layout,
  SVG,
} from 'twitch-core-ui';
import { ContrastLayout } from '../ContrastLayout';
import { PlayerOverlay } from '../PlayerOverlay';

// re-enables the button inside the overlay to accept taps/clicks
const ScControlContainer = styled.div`
  pointer-events: auto;
`;

export enum AssetLocation {
  Left = 'left',
  Right = 'right',
}

export interface PlayerControlsOverlayProps {
  asset: SVGAsset;
  assetLocation: AssetLocation;
  onTapOverlay: () => void;
  text: string;
}

/**
 * Overlay designed for showing a player state and allowing the user to tap to
 * interact with the player.
 */
export const PlayerControlsOverlayBase: FC<PlayerControlsOverlayProps> = ({
  asset,
  assetLocation,
  onTapOverlay,
  text,
}) => (
  <PlayerOverlay onTapOverlay={onTapOverlay}>
    <Layout
      alignItems={AlignItems.Start}
      display={Display.Flex}
      fullHeight
      fullWidth
      justifyContent={JustifyContent.Center}
      padding={{ x: 5 }}
    >
      <ScControlContainer>
        <ContrastLayout
          display={Display.Flex}
          flexDirection={
            assetLocation === AssetLocation.Right
              ? FlexDirection.Row
              : FlexDirection.RowReverse
          }
        >
          <CoreText color={Color.Overlay}>{text}</CoreText>
          <Layout
            alignItems={AlignItems.Center}
            alignSelf={AlignSelf.Start}
            display={Display.Flex}
            flexShrink={0}
            justifyContent={JustifyContent.Center}
            padding={
              assetLocation === AssetLocation.Right
                ? { left: 0.5 }
                : { right: 0.5 }
            }
          >
            <SVG asset={asset} />
          </Layout>
        </ContrastLayout>
      </ScControlContainer>
    </Layout>
  </PlayerOverlay>
);

PlayerControlsOverlayBase.displayName = 'PlayerControlsOverlayBase';

export const PlayerControlsOverlay = memo(PlayerControlsOverlayBase);
