import { VideoType, setMatureCookie } from 'pulsar';
import type { FormatMessage } from 'tachyon-intl';
import { getOverlaySrcAndSrcSet } from 'tachyon-more-ui';
import type { OverlayMessage } from '../OverlayManager';
import { PlayerOverlayType } from '../OverlayManager';

export const OFFLINE_OVERLAY_SIZES = [
  {
    mediaCondition: '(max-width: 960px) and (orientation: landscape)',
    size: '640px',
  },
  {
    mediaCondition: '(max-width: 1280px) and (orientation: landscape)',
    size: '960px',
  },
  {
    mediaCondition: '(max-width: 1600px) and (orientation: landscape)',
    size: '1280px',
  },
  { size: '100vw' },
];

export type ChannelOfflineOverlayOpts = {
  formatMessage: FormatMessage;
  offlineImageURL: string | null;
};

export function channelOfflineOverlay({
  formatMessage,
  offlineImageURL,
}: ChannelOfflineOverlayOpts): OverlayMessage {
  return {
    backgroundImage: {
      sizes: OFFLINE_OVERLAY_SIZES,
      srcAndSrcSet: getOverlaySrcAndSrcSet(offlineImageURL),
    },
    fatal: true,
    text: formatMessage('OFFLINE', 'PlayerOverlay'),
    type: PlayerOverlayType.Message,
  };
}

export function playbackErrorOverlay(
  formatMessage: FormatMessage,
  onRetryClick: () => void,
): OverlayMessage {
  return {
    button: {
      onClick: onRetryClick,
      text: formatMessage('Reload', 'PlayerOverlay'),
    },
    fatal: true,
    text: formatMessage(
      'An error occurred while playing this content.',
      'PlayerOverlay',
    ),
    type: PlayerOverlayType.Message,
  };
}

export function subOnlyLiveOverlay(
  formatMessage: FormatMessage,
  ownerDisplayName: string,
): OverlayMessage {
  return {
    fatal: true,
    text: formatMessage(
      '{ownerDisplayName} is streaming to subscribers.',
      { ownerDisplayName },
      'PlayerOverlay',
    ),
    type: PlayerOverlayType.Message,
  };
}

export function subOnlyVideoOverlay(
  formatMessage: FormatMessage,
): OverlayMessage {
  return {
    fatal: true,
    text: formatMessage(
      'This video is only available for subscribers.',
      'PlayerOverlay',
    ),
    type: PlayerOverlayType.Message,
  };
}

export type MatureContentOverlayOpts = {
  contentType: VideoType;
  formatMessage: FormatMessage;
  onAccept: () => void;
};

export function matureContentOverlay({
  contentType,
  formatMessage,
  onAccept,
}: MatureContentOverlayOpts): OverlayMessage {
  const text =
    contentType === VideoType.Stream
      ? formatMessage(
          'The broadcaster indicated that this channel is intended for mature audiences.',
          'PlayerOverlay',
        )
      : formatMessage(
          'The broadcaster indicated that this video is intended for mature audiences.',
          'PlayerOverlay',
        );

  return {
    button: {
      onClick: (): void => {
        setMatureCookie();
        onAccept();
      },
      text: formatMessage('Start Watching', 'PlayerOverlay'),
    },
    fatal: false,
    text,
    type: PlayerOverlayType.Message,
  };
}

export function regionalBlockOverlay(
  formatMessage: FormatMessage,
): OverlayMessage {
  return {
    fatal: true,
    text: formatMessage(
      'This premium content is not available in your region.',
      'PlayerOverlay',
    ),
    type: PlayerOverlayType.Message,
  };
}

export function unknownRestrictionOverlay(
  formatMessage: FormatMessage,
): OverlayMessage {
  return {
    fatal: true,
    text: formatMessage('This content is not available.', 'PlayerOverlay'),
    type: PlayerOverlayType.Message,
  };
}
