import { internet, lorem } from 'faker';
import { VideoType, lacksMatureCookie } from 'pulsar';
import { DefaultTwitchIntl } from 'tachyon-intl';
import { getOverlaySrcAndSrcSet } from 'tachyon-more-ui';
import { renderTachyonLink } from '../../../../routing';
import { PlayerOverlayType } from '../OverlayManager';
import {
  OFFLINE_OVERLAY_SIZES,
  channelOfflineOverlay,
  matureContentOverlay,
  playbackErrorOverlay,
  regionalBlockOverlay,
  subOnlyLiveOverlay,
  unknownRestrictionOverlay,
} from '.';

jest.mock('../../../../routing', () => ({
  ...jest.requireActual('../../../../routing'),
  renderTachyonLink: jest.fn(),
}));

jest.mock('tachyon-more-ui', () => ({
  ...jest.requireActual('tachyon-more-ui'),
  getOverlaySrcAndSrcSet: jest.fn(),
}));
const getOverlaySrcAndSrcSetMock = getOverlaySrcAndSrcSet as jest.Mock;

const formatMessage = DefaultTwitchIntl.formatMessage;

describe('overlay messages', () => {
  beforeEach(() => {
    (renderTachyonLink as jest.Mock).mockReset();
  });

  describe(matureContentOverlay, () => {
    it('returns correct values for streams', () => {
      expect(
        matureContentOverlay({
          contentType: VideoType.Stream,
          formatMessage,
          onAccept: jest.fn(),
        }),
      ).toEqual({
        button: {
          onClick: expect.any(Function),
          text: expect.any(String),
        },
        fatal: false,
        text: expect.stringContaining('channel'),
        type: PlayerOverlayType.Message,
      });
    });

    it('returns correct values for videos', () => {
      expect(
        matureContentOverlay({
          contentType: VideoType.Video,
          formatMessage,
          onAccept: jest.fn(),
        }),
      ).toEqual({
        button: {
          onClick: expect.any(Function),
          text: expect.any(String),
        },
        fatal: false,
        text: expect.stringContaining('video'),
        type: PlayerOverlayType.Message,
      });
    });

    it('creates an onClick handler that sets cookie and calls onAccept', () => {
      expect(lacksMatureCookie()).toBeTruthy();
      const onAccept = jest.fn();
      const overlay = matureContentOverlay({
        contentType: VideoType.Stream,
        formatMessage,
        onAccept,
      });
      overlay.button!.onClick!(undefined as any);
      expect(onAccept).toHaveBeenCalled();
      expect(lacksMatureCookie()).toBeFalsy();
    });
  });

  describe(regionalBlockOverlay, () => {
    it('returns correct values', () => {
      expect(regionalBlockOverlay(formatMessage)).toEqual({
        fatal: true,
        text: expect.any(String),
        type: PlayerOverlayType.Message,
      });
    });
  });

  describe(unknownRestrictionOverlay, () => {
    it('returns correct values', () => {
      expect(unknownRestrictionOverlay(formatMessage)).toEqual({
        fatal: true,
        text: expect.any(String),
        type: PlayerOverlayType.Message,
      });
    });
  });

  describe(channelOfflineOverlay, () => {
    it('returns correct values for offline image url', () => {
      const expectedSrcAndSrcSet = { src: lorem.words() };
      getOverlaySrcAndSrcSetMock.mockReturnValue(expectedSrcAndSrcSet);
      const offlineImageURL = internet.url();

      expect(channelOfflineOverlay({ formatMessage, offlineImageURL })).toEqual(
        {
          backgroundImage: {
            sizes: OFFLINE_OVERLAY_SIZES,
            srcAndSrcSet: expectedSrcAndSrcSet,
          },
          fatal: true,
          text: expect.any(String),
          type: PlayerOverlayType.Message,
        },
      );
      expect(getOverlaySrcAndSrcSet).toHaveBeenLastCalledWith(offlineImageURL);
    });
  });

  describe(playbackErrorOverlay, () => {
    it('returns correct values', () => {
      expect(playbackErrorOverlay(formatMessage, () => ({}))).toEqual({
        button: {
          onClick: expect.any(Function),
          text: 'Reload',
        },
        fatal: true,
        text: expect.any(String),
        type: PlayerOverlayType.Message,
      });
    });
  });

  describe(subOnlyLiveOverlay, () => {
    it('returns correct values', () => {
      const ownerDisplayName = lorem.word();
      expect(subOnlyLiveOverlay(formatMessage, ownerDisplayName)).toEqual({
        fatal: true,
        text: expect.any(String),
        type: PlayerOverlayType.Message,
      });
    });
  });
});
