import type { VideoStatus } from '../../VodPlayer';

export interface NextVideoNode {
  readonly id: string;
  readonly previewThumbnailURL: string;
  readonly status: VideoStatus | null;
  readonly title: string | null;
}

export interface NextVideoArgs {
  /**
   * A video ID to exclude as a next video recommendation typically because
   * it was the content that just ended.
   */
  omitVideoID: string | null;
  readonly recentVideos: ReadonlyArray<{
    readonly node: NextVideoNode | null;
  } | null> | null;
}

export interface NextVideo {
  id: string;
  previewThumbnailURL: string;
  title: string;
}

export function getNextVideo({
  omitVideoID,
  recentVideos,
}: NextVideoArgs): NextVideo | null {
  if (!recentVideos) {
    return null;
  }

  for (const edge of recentVideos) {
    // We skip the video if its status is "RECORDING" since it's incomplete and
    // will end unexpectedly and doesn't have a preview image yet.
    // Both of which make for a bad UX.
    if (
      edge?.node &&
      edge.node.id !== '' &&
      edge.node.id !== omitVideoID &&
      edge.node.status === 'RECORDED'
    ) {
      const nextVideo = edge.node;

      return {
        id: nextVideo.id,
        previewThumbnailURL: nextVideo.previewThumbnailURL,
        // If the VOD is valid, it has a title
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        title: nextVideo.title!,
      };
    }
  }

  return null;
}
