import type { FC } from 'react';
import { createContext, useContext, useMemo, useState } from 'react';
import { useStaticEnvironment } from 'tachyon-environment';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useEvent } from 'tachyon-utils';
import { isStandaloneApp } from '../../../../utils';

/**
 * Tracks successful PWA installations (A2HS). This is only reported by Chrome.
 */
export const APP_INSTALLED = 'app_installed';

type BeforeInstallPromptEvent = Event & { prompt: () => Promise<void> };

type AddToHomeScreenContext = {
  canInstall: boolean;
  dismissIosInstructions: () => void;
  install: () => void;
  showIosInstructions: boolean;
};

const addToHomeScreenContext = createContext<AddToHomeScreenContext>({
  canInstall: false,
  dismissIosInstructions: () => false,
  install: () => false,
  showIosInstructions: false,
});

export function useAddToHomeScreen(): AddToHomeScreenContext {
  return useContext(addToHomeScreenContext);
}

export const AddToHomeScreenRoot: FC = ({ children }) => {
  const {
    client: { agentInfo },
  } = useStaticEnvironment();

  // Chrome Only. First classed detection of PWA installability
  const [beforeInstallPromptEvent, setBeforeInstallPromptEvent] =
    useState<BeforeInstallPromptEvent>();

  const trackInteraction = useInteractionTracking();

  // Chrome Only
  useEvent('beforeinstallprompt', (e) => {
    // try to prevent the native prompt from showing
    // https://developers.google.com/web/updates/2018/06/a2hs-updates#what%E2%80%99s_changing
    e.preventDefault();
    setBeforeInstallPromptEvent(e as BeforeInstallPromptEvent);
  });

  // Chrome Only
  useEvent('appinstalled', () => {
    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: APP_INSTALLED,
    });
    // so installable state does not persist after installing
    setBeforeInstallPromptEvent(undefined);
  });

  const [showIosInstructions, setShowIosInstructions] = useState(false);

  // Apple enabled PWA capabilities in Safari 11.3.
  const isSupportedIosSafari =
    agentInfo.satisfies({ mobile: { safari: '>11.3' } }) ?? false;

  const canInstall =
    !isStandaloneApp() &&
    (beforeInstallPromptEvent !== undefined || isSupportedIosSafari);

  const ctx = useMemo(
    () => ({
      canInstall,
      dismissIosInstructions() {
        setShowIosInstructions(false);
      },
      install() {
        if (isSupportedIosSafari) {
          setShowIosInstructions(true);
        } else {
          beforeInstallPromptEvent?.prompt();
        }
      },
      showIosInstructions,
    }),
    [
      beforeInstallPromptEvent,
      canInstall,
      isSupportedIosSafari,
      showIosInstructions,
      setShowIosInstructions,
    ],
  );

  return <addToHomeScreenContext.Provider children={children} value={ctx} />;
};
AddToHomeScreenRoot.displayName = 'AddToHomeScreenRoot';
