import { act, renderHook } from '@testing-library/react-hooks';
import { getOtherEnvMock, useStaticEnvironment } from 'tachyon-environment';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { useEvent } from 'tachyon-utils';
import { isStandaloneApp } from '../../../../utils';
import { APP_INSTALLED, AddToHomeScreenRoot, useAddToHomeScreen } from '.';

jest.mock('../../../../utils', () => ({
  isStandaloneApp: jest.fn(() => false),
}));
const mockIsStandaloneApp = isStandaloneApp as jest.Mock;

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useStaticEnvironment: jest.fn(),
}));
const mockUseStaticEnvironment = useStaticEnvironment as jest.Mock;
mockUseStaticEnvironment.mockImplementation(() => getOtherEnvMock());

jest.mock('tachyon-utils', () => ({
  ...jest.requireActual('tachyon-utils'),
  useEvent: jest.fn(),
}));
const mockUseEvent = useEvent as jest.Mock;

jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  useInteractionTracking: jest.fn(),
}));
const mockUseInteractionTracking = useInteractionTracking as jest.Mock;

describe(AddToHomeScreenRoot, () => {
  const setup = createMountWrapperFactory(AddToHomeScreenRoot);

  it('tracks appinstalled event', () => {
    const trackInteraction = jest.fn();
    mockUseInteractionTracking.mockImplementationOnce(() => trackInteraction);
    const events: any = {};
    mockUseEvent.mockImplementation((event, handler) => {
      events[event] = handler;
    });

    setup();
    events['appinstalled']();

    expect(trackInteraction).toHaveBeenCalledWith({
      interaction: InteractionType.Click,
      interactionContent: APP_INSTALLED,
    });
  });
});

describe(useAddToHomeScreen, () => {
  describe('canInstall', () => {
    it('false when in standalone (A2HS) context', () => {
      mockIsStandaloneApp.mockImplementationOnce(() => true);
      const { result } = renderHook(useAddToHomeScreen, {
        wrapper: AddToHomeScreenRoot,
      });
      expect(result.current.canInstall).toEqual(false);
    });

    it('true when beforeInstallPromptEvent present (all browsers except iOS)', () => {
      const events: any = {};
      mockUseEvent.mockImplementation((event, handler) => {
        events[event] = handler;
      });
      const { result } = renderHook(useAddToHomeScreen, {
        wrapper: AddToHomeScreenRoot,
      });

      expect(result.current.canInstall).toEqual(false);
      act(() => {
        events['beforeinstallprompt']({ preventDefault: jest.fn() });
      });
      expect(result.current.canInstall).toEqual(true);
    });

    it('true when Safari 11.3+', () => {
      const satisfies = jest.fn(() => true);
      mockUseStaticEnvironment.mockImplementationOnce(() => ({
        client: { agentInfo: { satisfies } },
      }));
      const { result } = renderHook(useAddToHomeScreen, {
        wrapper: AddToHomeScreenRoot,
      });

      expect(satisfies).toHaveBeenCalledWith({ mobile: { safari: '>11.3' } });
      expect(result.current.canInstall).toEqual(true);
    });
  });

  describe('install', () => {
    it('shows iosInstructions when iOS', () => {
      const satisfies = jest.fn(() => true);
      mockUseStaticEnvironment.mockImplementationOnce(() => ({
        client: { agentInfo: { satisfies } },
      }));

      const { result } = renderHook(useAddToHomeScreen, {
        wrapper: AddToHomeScreenRoot,
      });

      expect(result.current.showIosInstructions).toEqual(false);
      act(() => {
        result.current.install();
      });
      expect(result.current.showIosInstructions).toEqual(true);
    });

    it('invokes beforeInstallPromptEvent.prompt when not iOS', () => {
      const events: any = {};
      mockUseEvent.mockImplementation((event, handler) => {
        events[event] = handler;
      });

      const { result } = renderHook(useAddToHomeScreen, {
        wrapper: AddToHomeScreenRoot,
      });

      const mockPrompt = jest.fn();
      act(() => {
        events['beforeinstallprompt']({
          preventDefault: jest.fn(),
          prompt: mockPrompt,
        });
      });

      act(() => {
        result.current.install();
      });
      expect(mockPrompt).toHaveBeenCalledTimes(1);
    });
  });
});
