import type { FC } from 'react';
import { useEffect, useState } from 'react';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { isBrowser, useStorage } from 'tachyon-utils';
import { useAddToHomeScreen } from '../AddToHomeScreenRoot';
import { AddToHomescreenPrompt, IOSInstallInstructions } from '../Callouts';

/**
 * Tracks showing the install prompt for PWA (A2HS).
 */
const APP_INSTALL_PROMPT = 'app_install_prompt';

/**
 * Tracks clicks of 'Install Web App' button on the install prompt for PWA (A2HS).
 */
const APP_INSTALL_PROMPT_INSTALL = 'app_install_prompt_install';

/**
 * Tracks dismissals of the install prompt for PWA (A2HS).
 */
const APP_INSTALL_PROMPT_DISMISSED = 'app_install_prompt_dismissed';

/**
 * Tracks dismissals not from clicking the close button such as clicking outside of the CTA.
 */
const APP_INSTALL_PROMPT_DISMISSED_IMPLICIT =
  'app_install_prompt_dismissed_implicit';

type AddToHomeScreenStorage = {
  shouldShowCTA: boolean;
};

type AddToHomeScreen = {
  disableShowCTA: () => void;
  enableShowCTA: () => void;
  shouldShowCTA: boolean;
};

const ADD_TO_HOMESCREEN_CTA_STORAGE_KEY = 'add-to-home-screen-cta';

// istanbul ignore next: trivial
export function useAddToHomeScreenCTA(): AddToHomeScreen {
  const [canShowCTA, setCanShowCTA] = useStorage<AddToHomeScreenStorage>(
    ADD_TO_HOMESCREEN_CTA_STORAGE_KEY,
    { shouldShowCTA: true },
  );

  return {
    disableShowCTA() {
      setCanShowCTA({ shouldShowCTA: false });
    },
    enableShowCTA() {
      setCanShowCTA({ shouldShowCTA: true });
    },
    shouldShowCTA: canShowCTA.shouldShowCTA,
  };
}

/**
 * Wait for a session with 5 page views before showing the CTA for the first time.
 */
export const PAGEVIEW_THRESHOLD = 5;

/**
 * A call to action for iOS and Android devices prompting the user to add to home screen (A2HS).
 *
 * Rules:
 *
 * - We wait for a session with 5 pageviews before showing the CTA for the first time.
 *
 * - After being dismissed (either by ignoring the CTA or by explicitly closing it) we won't show the CTA again until the user tries to chat or follow.
 *
 * Caveats:
 *
 * - Tracking will be tied to the device, so a user with multiple devices would see the CTA semantics above on each device.
 *
 * - Because we don't have server side storage, if the user clears their local storage we will show the CTA again after 5 pageviews. Safari cleans up storage automatically (I think after a week of not visiting the page?) so infrequent iOS users may see this CTA multiple times without trying to follow or chat.
 *
 */
export const AddToHomeScreenCTA: FC = () => {
  const { canInstall, dismissIosInstructions, install, showIosInstructions } =
    useAddToHomeScreen();
  const [showCTA, setShowCTA] = useState(false);
  const { disableShowCTA, shouldShowCTA } = useAddToHomeScreenCTA();
  const {
    common: { platform },
  } = useStaticEnvironment();
  const trackInteraction = useInteractionTracking();
  const { pageCount } = useRouterUtils();

  const canShowCTA =
    canInstall &&
    shouldShowCTA &&
    /* The strict check (=== PAGEVIEW_THRESHOLD as opposed to >= PAGEVIEW_THRESHOLD) is intentional so that we only show banner once per app session. When triggering an upsell action, the upsell won't show in the current session, but will on the subsequent session. */
    pageCount === PAGEVIEW_THRESHOLD &&
    platform === Platform.MobileWeb;

  useEffect(() => {
    if (!showCTA && canShowCTA) {
      setShowCTA(true);
      disableShowCTA();
      trackInteraction({
        interaction: InteractionType.Impression,
        interactionContent: APP_INSTALL_PROMPT,
      });
    }
  }, [disableShowCTA, showCTA, canShowCTA, trackInteraction]);

  if (!isBrowser()) {
    return null;
  }

  function closeCTA() {
    setShowCTA(false);

    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: APP_INSTALL_PROMPT_DISMISSED,
    });
  }

  function outsideClickCloseCTA() {
    setShowCTA(false);

    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: APP_INSTALL_PROMPT_DISMISSED_IMPLICIT,
    });
  }

  function installCTA() {
    setShowCTA(false);

    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: APP_INSTALL_PROMPT_INSTALL,
    });

    install();
  }

  return (
    <>
      <AddToHomescreenPrompt
        onClose={closeCTA}
        onInstall={installCTA}
        onOutsideClose={outsideClickCloseCTA}
        show={showCTA}
      />
      <IOSInstallInstructions
        onClose={dismissIosInstructions}
        show={showIosInstructions}
      />
    </>
  );
};
AddToHomeScreenCTA.displayName = 'AddToHomeScreenCTA';
