import type { FC } from 'react';
import styled from 'styled-components';
import { isIOS, useStaticEnvironment } from 'tachyon-environment';
import type { FormatMessage } from 'tachyon-intl';
import { useIntl } from 'tachyon-intl';
import { IOSAddToHomescreenIcon, IOSShareIcon } from 'tachyon-more-ui';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import {
  AlignItems,
  BorderRadius,
  ButtonType,
  Callout,
  CalloutActions,
  CalloutMessage,
  CalloutOrientation,
  CoreImage,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  Modal,
  ModalSize,
  NotificationType,
  PillType,
  Position,
  Snackbar,
  SnackbarActions,
  TransitionType,
} from 'twitch-core-ui';
import { TOP_NAV_HEIGHT_PX } from '../../../../config';
import { RouteName, routeNameFromPathname } from '../../../../routing';
import { OpenInApp } from '../../OpenInApp';

type CTAProps = {
  onClose: () => void;
  onInstall: () => void;
  onOutsideClose: () => void;
  show: boolean;
};

// istanbul ignore next: trivial
function close(formatMessage: FormatMessage): string {
  return formatMessage('Close', 'AddToHomeScreenCTA');
}

// istanbul ignore next: trivial
export const IOSInstallInstructions: FC<Pick<CTAProps, 'onClose' | 'show'>> = ({
  onClose,
  show,
}) => {
  const { formatMessage } = useIntl();

  return (
    <Modal
      backdrop={null}
      backdropTransitionType={TransitionType.None}
      borderRadius={BorderRadius.None}
      onRequestClose={onClose}
      show={show}
      size={ModalSize.Small}
      transitionType={TransitionType.TranslateBottom}
    >
      <Layout attachBottom attachLeft attachRight position={Position.Absolute}>
        <Callout
          closeButton={{
            'aria-label': close(formatMessage),
            onClick: onClose,
          }}
          message={
            <>
              <CalloutMessage
                title={formatMessage(
                  'Add Twitch to Home Screen',
                  'AddToHomeScreenCTA',
                )}
              />
              <Layout padding={{ top: 1 }}>
                {formatMessage(
                  "It's easy to add a shortcut to Twitch on your iOS device's home screen!",
                  'AddToHomeScreenCTA',
                )}
                <Layout
                  alignItems={AlignItems.Center}
                  display={Display.Flex}
                  padding={{ top: 2 }}
                >
                  <Layout display={Display.Flex} padding={{ right: 1 }}>
                    <IOSShareIcon />
                  </Layout>
                  <CoreText bold>
                    {formatMessage(
                      "1) Tap the share button on Safari's menu bar",
                      'AddToHomeScreenCTA',
                    )}
                  </CoreText>
                </Layout>
                <Layout
                  alignItems={AlignItems.Center}
                  display={Display.Flex}
                  padding={{ top: 1 }}
                >
                  <Layout display={Display.Flex} padding={{ right: 1 }}>
                    <IOSAddToHomescreenIcon />
                  </Layout>
                  <CoreText bold>
                    {formatMessage(
                      "2) Tap 'Add to Home Screen'",
                      'AddToHomeScreenCTA',
                    )}
                  </CoreText>
                </Layout>
              </Layout>
            </>
          }
          orientation={CalloutOrientation.Column}
        />
      </Layout>
    </Modal>
  );
};
IOSInstallInstructions.displayName = 'IOSInstallInstructions';

export const AddToHomescreenPrompt: FC<CTAProps> = ({ show, ...props }) => {
  const { currentPathname } = useRouterUtils();
  const currentRouteName = routeNameFromPathname(currentPathname);

  const showFullCTA =
    show &&
    currentRouteName !== RouteName.Channel &&
    window.matchMedia('(min-height: 500px)').matches;

  const showMinimalCTA = show && !showFullCTA;

  return (
    <>
      <FullCTA show={showFullCTA} {...props} />
      <MinimalCTA show={showMinimalCTA} {...props} />
    </>
  );
};

AddToHomescreenPrompt.displayName = 'AddToHomescreenPrompt';

// istanbul ignore next: trivial
export const FullCTA: FC<CTAProps> = ({
  onClose,
  onInstall,
  onOutsideClose,
  show,
}) => {
  const { formatMessage } = useIntl();
  const { client } = useStaticEnvironment();
  const isIos = isIOS(client.agentInfo);

  const srcAndSrcSet = isIos
    ? {
        src: '/static/images/add-to-home-screen-cta-ios.jpg',
        srcSet: '/static/images/add-to-home-screen-cta-ios.jpg 1x',
      }
    : {
        src: '/static/images/add-to-home-screen-cta-android.jpg',
        srcSet: '/static/images/add-to-home-screen-cta-android.jpg 1x',
      };

  return (
    <Modal
      backdrop={null}
      backdropTransitionType={TransitionType.None}
      borderRadius={BorderRadius.None}
      onRequestClose={onOutsideClose}
      show={show}
      size={ModalSize.Small}
      transitionType={TransitionType.TranslateBottom}
    >
      <Layout
        attachBottom
        attachLeft
        attachRight
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        position={Position.Absolute}
      >
        <Callout
          actions={
            <CalloutActions
              primaryButton={{
                children: formatMessage(
                  'Install Web App',
                  'AddToHomeScreenCTA',
                ),
                onClick: onInstall,
              }}
              secondaryButton={{
                children: (
                  <OpenInApp
                    callToActionIdentifier="add_to_home_screen_cta"
                    variant={ButtonType.Secondary}
                  >
                    {isIos
                      ? formatMessage('Open iOS App', 'AddToHomeScreenCTA')
                      : formatMessage('Open Android App', 'AddToHomeScreenCTA')}
                  </OpenInApp>
                ),
                type: ButtonType.Text,
              }}
            />
          }
          closeButton={{
            'aria-label': close(formatMessage),
            onClick: onClose,
          }}
          message={
            <>
              <CalloutMessage
                description={formatMessage(
                  'Installing uses very little space and provides a richer experience.',
                  'AddToHomeScreenCTA',
                )}
                pill={{
                  label: formatMessage('NEW', 'AddToHomeScreenCTA'),
                  type: PillType.Brand,
                }}
                title={formatMessage(
                  'Install the Web App',
                  'AddToHomeScreenCTA',
                )}
              />
              <Layout fullWidth padding={{ top: 1 }}>
                <CoreImage alt="" {...srcAndSrcSet} />
              </Layout>
            </>
          }
          orientation={CalloutOrientation.Column}
        />
      </Layout>
    </Modal>
  );
};
FullCTA.displayName = 'FullCTA';

// istanbul ignore next: trivial
const ScMinimalCTAContainer = styled(Layout)`
  margin-top: calc(${TOP_NAV_HEIGHT_PX}px + 2rem);
`;

// istanbul ignore next: trivial
export const MinimalCTA: FC<CTAProps> = ({
  onClose,
  onInstall,
  onOutsideClose,
  show,
}) => {
  const { formatMessage } = useIntl();

  return (
    <Modal
      backdrop={null}
      backdropTransitionType={TransitionType.None}
      borderRadius={BorderRadius.None}
      onRequestClose={onOutsideClose}
      show={show}
      size={ModalSize.Small}
      transitionType={TransitionType.TranslateTop}
    >
      <ScMinimalCTAContainer
        attachLeft
        attachRight
        attachTop
        display={Display.Flex}
        justifyContent={JustifyContent.Center}
        position={Position.Absolute}
      >
        <Snackbar
          actions={
            <SnackbarActions
              primaryButton={{
                children: formatMessage('Install', 'AddToHomeScreenCTA'),
                onClick: onInstall,
              }}
              type={NotificationType.Info}
            />
          }
          closeButton={{
            'aria-label': close(formatMessage),
            onClick: onClose,
          }}
          message={formatMessage(
            'Get a lightweight Twitch for faster fun.',
            'AddToHomeScreenCTA',
          )}
          type={NotificationType.Info}
        />
      </ScMinimalCTAContainer>
    </Modal>
  );
};
MinimalCTA.displayName = 'MinimalCTA';
