import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { getChannelThumbnailSrcAndSrcSet } from 'tachyon-more-ui';
import { exhaustedCase } from 'tachyon-utils';
import { CoreImage, CoreLink, CoreLinkType } from 'twitch-core-ui';
import type { TwitchIntl } from 'twitch-intl';
import { RouteName, renderTachyonLink } from '../../../routing';
import { StreamStatusIndicator } from '../StreamStatusIndicator';
import { FeaturedCardCommon } from './FeaturedCardCommon';
import type { ChannelFeaturedCardStream_stream } from './__generated__/ChannelFeaturedCardStream_stream.graphql';

type ChannelFeaturedCardStreamType = 'hosting' | 'live';

type ChannelFeaturedCardStreamProps = {
  stream: ChannelFeaturedCardStream_stream;
  type: ChannelFeaturedCardStreamType;
};

// istanbul ignore next: trivial
const getHeader = (
  { formatMessage }: TwitchIntl,
  type: ChannelFeaturedCardStreamType,
) => {
  switch (type) {
    case 'live':
      return formatMessage('Live', 'ChannelFeaturedCardStream');
    case 'hosting':
      return formatMessage("While I'm Offline...", 'ChannelFeaturedCardStream');
    default:
      return exhaustedCase(type, '');
  }
};

// istanbul ignore next: trivial
export const ChannelFeaturedCardStreamBase: FC<ChannelFeaturedCardStreamProps> =
  ({ stream, type }) => {
    const intl = useIntl();

    if (!stream.broadcaster) {
      return null;
    }

    return (
      <CoreLink
        linkTo="/deferToRenderLink"
        renderLink={renderTachyonLink({
          route: RouteName.Channel,
          routeParams: { login: stream.broadcaster.login },
        })}
        variant={CoreLinkType.Inherit}
      >
        <FeaturedCardCommon
          gameLabel={stream.game?.displayName ?? ''}
          header={getHeader(intl, type)}
          image={
            <CoreImage
              {...getChannelThumbnailSrcAndSrcSet(stream.previewImageURL)}
              alt=""
            />
          }
          title={stream.broadcaster.broadcastSettings?.title ?? ''}
          topLeft={<StreamStatusIndicator isOverlay type={type} />}
        />
      </CoreLink>
    );
  };

ChannelFeaturedCardStreamBase.displayName = 'ChannelFeaturedCardStream';

export const ChannelFeaturedCardStream = createFragmentContainer(
  ChannelFeaturedCardStreamBase,
  {
    stream: graphql`
      fragment ChannelFeaturedCardStream_stream on Stream {
        id
        broadcaster {
          broadcastSettings {
            title
          }
        }
        previewImageURL
        broadcaster {
          login
        }
        game {
          displayName
        }
      }
    `,
  },
);
