import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  formatTotalViewCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { exhaustedCase } from 'tachyon-utils';
import {
  CoreImage,
  CoreLink,
  CoreLinkType,
  MediaCardStat,
} from 'twitch-core-ui';
import type { TwitchIntl } from 'twitch-intl';
import { RouteName, renderTachyonLink } from '../../../routing';
import { FeaturedCardCommon } from './FeaturedCardCommon';
import type { ChannelFeaturedCardVideo_video } from './__generated__/ChannelFeaturedCardVideo_video.graphql';

type ChannelFeaturedCardVideoType = 'past_broadcast' | 'recent_highlight';

type ChannelFeaturedCardVideoProps = {
  type: ChannelFeaturedCardVideoType;
  video: ChannelFeaturedCardVideo_video;
};

// istanbul ignore next: trivial
const getHeader = (
  { formatMessage }: TwitchIntl,
  type: ChannelFeaturedCardVideoType,
) => {
  switch (type) {
    case 'past_broadcast':
      return formatMessage('Past Broadcast', 'ChannelFeaturedCardVideo');
    case 'recent_highlight':
      return formatMessage('Recent Highlight', 'ChannelFeaturedCardVideo');
    default:
      return exhaustedCase(type, '');
  }
};

// istanbul ignore next: trivial
export const ChannelFeaturedCardVideoBase: FC<ChannelFeaturedCardVideoProps> =
  ({ type, video }) => {
    const intl = useIntl();
    const { formatMessage, formatPastRelativeDate } = intl;
    const gameName = video.game?.name ?? '';
    const publishedAt = video.publishedAt;

    const gameLabel = publishedAt
      ? `${gameName} · ${formatPastRelativeDate(new Date(publishedAt))}`
      : gameName;

    return (
      <CoreLink
        linkTo="/deferToRenderLink"
        renderLink={renderTachyonLink({
          route: RouteName.Vod,
          routeParams: { videoID: video.id },
        })}
        variant={CoreLinkType.Inherit}
      >
        <FeaturedCardCommon
          gameLabel={gameLabel}
          header={getHeader(intl, type)}
          image={
            <CoreImage
              {...getChannelThumbnailSrcAndSrcSet(video.previewThumbnailURL)}
              alt=""
            />
          }
          title={video.title ?? ''}
          topLeft={
            video.viewCount !== null ? (
              <MediaCardStat>
                {formatTotalViewCount(formatMessage, video.viewCount)}
              </MediaCardStat>
            ) : null
          }
        />
      </CoreLink>
    );
  };

ChannelFeaturedCardVideoBase.displayName = 'ChannelFeaturedCardVideo';

export const ChannelFeaturedCardVideo = createFragmentContainer(
  ChannelFeaturedCardVideoBase,
  {
    video: graphql`
      fragment ChannelFeaturedCardVideo_video on Video {
        id
        title
        viewCount
        previewThumbnailURL
        publishedAt
        game {
          name
        }
      }
    `,
  },
);
