import type { FC, MouseEvent } from 'react';
import { memo, useState } from 'react';
import styled from 'styled-components';
import {
  ExtendInteractionMedium,
  InteractionType,
  useInteractionTracking,
} from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Background,
  BorderRadius,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  Layout,
} from 'twitch-core-ui';
import { useAddToHomeScreenCTA } from '../../AddToHomeScreen';
import { UpsellSheet } from '../../UpsellSheet';

const INTERACTION_MEDIUM = 'chat_input_upsell';
const BRANCH_CAMPAIGN = 'chat_input_upsell';

const ScSpan = styled.span`
  font-weight: bold;
`;

export type ChatInputProps = {
  channelDisplayName: string;
};

export const ChatInputUpsellBase: FC<ChatInputProps> = ({
  channelDisplayName,
}) => {
  const trackInteraction = useInteractionTracking();
  const { formatMessage } = useIntl();
  const { enableShowCTA } = useAddToHomeScreenCTA();

  const [chatUpsellOpen, setChatUpsellOpen] = useState(false);

  const upsellTitle = formatMessage(
    'Open the app to chat with {displayName}!',
    {
      displayName: <ScSpan>{channelDisplayName}</ScSpan>,
    },
    'ChatInputUpsell',
  );
  const upsellDescription = formatMessage(
    "Level up your experience! Participate with {displayName}'s chat and the community in the app.",
    {
      displayName: <ScSpan>{channelDisplayName}</ScSpan>,
    },
    'ChatInputUpsell',
  );

  const upsellButtonText = formatMessage('Open App', 'ChatInputUpsell');

  function onClick(e: MouseEvent<HTMLElement>): void {
    e.stopPropagation();
    setChatUpsellOpen(true);
    enableShowCTA();
    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: 'chat_input',
    });
  }

  return (
    <ExtendInteractionMedium value={INTERACTION_MEDIUM}>
      <Interactable onClick={onClick}>
        <Layout
          alignItems={AlignItems.Center}
          background={Background.Alt}
          border
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
          padding={1}
        >
          <Layout flexGrow={1}>
            <CoreText color={Color.Alt2}>
              {formatMessage('Send a message', 'ChatInput')}
            </CoreText>
          </Layout>
          <Layout
            background={Background.Accent}
            borderRadius={BorderRadius.Small}
            padding={{ x: 1, y: 0.5 }}
          >
            <CoreText color={Color.Overlay} fontSize={FontSize.Size7}>
              {formatMessage('Chat', 'ChatInput')}
            </CoreText>
          </Layout>
        </Layout>
      </Interactable>
      {chatUpsellOpen && (
        <UpsellSheet
          branchCampaign={BRANCH_CAMPAIGN}
          buttonText={upsellButtonText}
          callToActionIdentifier="chat_input"
          description={upsellDescription}
          onClose={() => {
            setChatUpsellOpen(false);
          }}
          title={upsellTitle}
        />
      )}
    </ExtendInteractionMedium>
  );
};

ChatInputUpsellBase.displayName = 'ChatInputUpsell';

// needs to be memoized because it otherwise re-renders on every chat message...
export const ChatInputUpsell = memo(ChatInputUpsellBase);
