import type { UIEvent } from 'react';
import { Component } from 'react';
import styled from 'styled-components';
import type { ChatProps } from 'tachyon-chat-ui';
import { ChatMessage, withChat } from 'tachyon-chat-ui';
import type { StaticEnvironmentProps } from 'tachyon-environment';
import { isAndroidOS, isIOS, withStaticEnvironment } from 'tachyon-environment';
import type { IntlProps } from 'tachyon-intl';
import { withIntl } from 'tachyon-intl';
import type { CustomLatencyProps } from 'tachyon-latency-tracker';
import { withCustomLatency } from 'tachyon-latency-tracker';
import {
  Background,
  Button,
  ButtonType,
  Layout,
  Overflow,
  Position,
} from 'twitch-core-ui';
import { ChatInputUpsell } from './ChatInputUpsell';

export type ChatPaneProps = ChatProps &
  CustomLatencyProps &
  IntlProps &
  StaticEnvironmentProps & {
    channelDisplayName: string;
    preventInputUpsell?: true;
  };

type ChatState = {
  autoscroll: boolean;
};

const AUTOSCROLL_TRIGGER_ZONE_SIZE = 10;

/**
 * "height" is required to allow overflow enforcement.
 */
const ScMessageList = styled.ul`
  height: 100%;
  overflow-y: scroll;
`;

export class ChatPaneBase extends Component<ChatPaneProps, ChatState> {
  public static displayName = 'ChatPane';

  public chatList: HTMLUListElement | null = null;
  public userScroll: boolean;
  private reportChatConnected: () => void;

  constructor(props: ChatPaneProps) {
    super(props);
    this.userScroll = true;
    this.state = {
      autoscroll: true,
    };
    this.reportChatConnected = this.props.reportLatencyEvent('chatConnected');
  }

  public componentDidUpdate(): void {
    if (this.state.autoscroll) {
      this.scrollToBottom();
    }
  }

  public handleScroll = (event: UIEvent<HTMLUListElement>): void => {
    event.stopPropagation();

    // differentiate between user scoll and auto scroll
    if (!this.userScroll) {
      this.userScroll = true;
    } else if (
      this.chatList &&
      this.chatList.scrollTop + this.chatList.clientHeight >=
        this.chatList.scrollHeight - AUTOSCROLL_TRIGGER_ZONE_SIZE
    ) {
      if (!this.state.autoscroll) {
        this.setState({ autoscroll: true });
        this.props.unpause();
      }
    } else if (this.state.autoscroll) {
      this.setState({ autoscroll: false });
      this.props.pause();
    }
  };

  public scrollToBottom = (): void => {
    this.userScroll = false;
    if (this.chatList) {
      this.chatList.scrollTop = this.chatList.scrollHeight;
    }
  };

  public render(): JSX.Element {
    const {
      channelDisplayName,
      chatMessages,
      intl,
      preventInputUpsell,
      staticEnv: {
        client: { agentInfo },
      },
    } = this.props;
    const { formatMessage } = intl;
    const showInputUpsell =
      !preventInputUpsell && (isAndroidOS(agentInfo) || isIOS(agentInfo));

    if (chatMessages.length > 0) {
      this.reportChatConnected();
    }

    return (
      <>
        <Layout
          fullHeight
          overflow={Overflow.Hidden}
          position={Position.Relative}
        >
          <ScMessageList
            onScroll={this.handleScroll}
            ref={(c) => (this.chatList = c as HTMLUListElement)}
          >
            {chatMessages.map((message) => (
              <ChatMessage intl={intl} key={message.id} message={message} />
            ))}
          </ScMessageList>
          {!this.state.autoscroll && (
            <Layout
              attachBottom
              background={Background.Overlay}
              fullWidth
              padding={0.5}
              position={Position.Absolute}
            >
              <Button
                fullWidth
                onClick={() => {
                  this.setState({ autoscroll: true }, this.scrollToBottom);
                  this.props.unpause();
                }}
                overlay
                variant={ButtonType.Text}
              >
                {formatMessage('More messages below', 'ChatPane')}
              </Button>
            </Layout>
          )}
        </Layout>
        {showInputUpsell && (
          <Layout flexShrink={0}>
            <ChatInputUpsell channelDisplayName={channelDisplayName} />
          </Layout>
        )}
      </>
    );
  }
}

export const ChatPane = withStaticEnvironment(
  withChat(withIntl(withCustomLatency(ChatPaneBase))),
);
