import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  Avatar,
  formatTotalViewCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { formatVideoLength, getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  CoreLink,
  CoreLinkType,
  MediaCard,
  MediaCardImage,
  MediaCardLink,
  MediaCardMeta,
  MediaCardStat,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../routing';
import type { ClipCard_clip } from './__generated__/ClipCard_clip.graphql';

type ClipCardProps = {
  clip: ClipCard_clip;
};

// istanbul ignore next: trivial
export const ClipCardBase: FC<ClipCardProps> = ({ clip }) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();

  if (!clip.broadcaster) {
    return null;
  }

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      renderLink={renderTachyonLink({
        route: RouteName.Clip,
        routeParams: { login: clip.broadcaster.login, slug: clip.slug },
      })}
      variant={CoreLinkType.Inherit}
    >
      <MediaCard
        image={
          <MediaCardImage
            bottomLeft={
              clip.viewCount ? (
                <MediaCardStat>
                  {formatTotalViewCount(
                    formatMessage,
                    clip.viewCount,
                    formatNumberShort(clip.viewCount, {
                      maximumSignificantDigits: 2,
                    }),
                  )}
                </MediaCardStat>
              ) : null
            }
            bottomRight={
              clip.createdAt ? (
                <MediaCardStat>
                  {formatPastRelativeDate(new Date(clip.createdAt))}
                </MediaCardStat>
              ) : null
            }
            image={
              <CardImage
                {...getChannelThumbnailSrcAndSrcSet(clip.thumbnailURL)}
                alt=""
              />
            }
            ratio={AspectRatio.Aspect16x9}
            topLeft={
              clip.durationSeconds ? (
                <MediaCardStat>
                  {formatVideoLength(clip.durationSeconds)}
                </MediaCardStat>
              ) : null
            }
          />
        }
        meta={
          <MediaCardMeta
            image={
              <Avatar
                alt=""
                sizeRem={3.6}
                src={clip.broadcaster.profileImageURL}
                userLogin={clip.broadcaster.login}
              />
            }
            links={
              <>
                <MediaCardLink>{clip.title}</MediaCardLink>
                {clip.game?.displayName ? (
                  <MediaCardLink>{clip.game.displayName}</MediaCardLink>
                ) : null}
              </>
            }
            title={getFormattedUserDisplayName(clip.broadcaster)}
          />
        }
      />
    </CoreLink>
  );
};

ClipCardBase.displayName = 'ClipCard';

export const ClipCard = createFragmentContainer(ClipCardBase, {
  clip: graphql`
    fragment ClipCard_clip on Clip {
      id
      broadcaster {
        login
        displayName
        profileImageURL(width: 50)
      }
      createdAt
      durationSeconds
      game {
        displayName
      }
      slug
      thumbnailURL(width: 480, height: 272)
      title
      viewCount
    }
  `,
});
