import type { FC, ReactChild } from 'react';
import { useEffect, useState } from 'react';
import { useDynamicSettings } from 'tachyon-dynamic-settings';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { acceptCookies, needsToAcceptCookies } from 'tachyon-page-utils';
import { useRequestInfo } from 'tachyon-relay';
import {
  Button,
  CoreLink,
  CoreText,
  FontSize,
  Layout,
  Modal,
  ModalBody,
  ModalSize,
  TextAlign,
} from 'twitch-core-ui';
import type { TomorrowDynamicSettings } from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';

const COOKIE_URL = 'https://www.twitch.tv/p/cookie-policy';

/**
 * Tracks showing the cookie notice
 */
const COOKIE_NOTICE_SHOW = 'cookie_notice_show';

/**
 * Tracks accepting the cookie notice
 */
const COOKIE_NOTICE_ACCEPT = 'cookie_notice_accept';

// istanbul ignore next: trivial
const CookieLink = (children: ReactChild) => (
  <CoreLink
    children={children}
    linkTo="/deferToRenderLink"
    renderLink={renderTachyonLink({
      route: RouteName.External,
      routeParams: { href: COOKIE_URL },
    })}
  />
);
CookieLink.displayName = 'CookieLink';

/**
 * A cookie banner that is shown once per year to mobile web users.
 *
 * Product spec: https://docs.google.com/document/d/1WwmtOAI9LvC8Sl2U2mLLqL5BP0Ct-2ieiMCNMOsgU9g/edit#
 *
 * To simplify the logic we show the notice every year, rather than trying to calculate 13 months in seconds (which would vary based on the month). Additionally, this modal is show to _all_ users, rather than trying to determine regions where it is required.
 */
// istanbul ignore next: trivial
export const CookieNotice: FC = () => {
  const { formatMessage } = useIntl();
  const {
    common: { platform },
  } = useStaticEnvironment();
  const { cookieNoticeCountryCodes } =
    useDynamicSettings<TomorrowDynamicSettings>();
  const requestInfo = useRequestInfo();
  const [showCookieNotice, setShowCookieNotice] = useState(false);
  const trackInteraction = useInteractionTracking();

  useEffect(() => {
    const cookieNoticeApplicable =
      platform !== Platform.Woodstock &&
      cookieNoticeCountryCodes.includes(requestInfo?.countryCode ?? '');

    if (cookieNoticeApplicable && needsToAcceptCookies()) {
      setShowCookieNotice(true);

      trackInteraction({
        interaction: InteractionType.Impression,
        interactionContent: COOKIE_NOTICE_SHOW,
      });
    }
  }, [platform, cookieNoticeCountryCodes, requestInfo, trackInteraction]);

  function acceptCookieNotice() {
    acceptCookies();
    setShowCookieNotice(false);

    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: COOKIE_NOTICE_ACCEPT,
    });
  }

  return (
    <Modal
      onRequestClose={() => false}
      // the modal can only be closed by accepting the coookie notice
      show={showCookieNotice}
      size={ModalSize.Small}
    >
      <ModalBody padding={{ x: 2, y: 2 }}>
        <Layout textAlign={TextAlign.Center}>
          <CoreText bold fontSize={FontSize.Size4}>
            {formatMessage('We value your privacy.', 'CookieNotice')}
          </CoreText>
        </Layout>
        <Layout padding={{ top: 1 }} textAlign={TextAlign.Center}>
          {formatMessage(
            'We use technologies such as <x:link>cookies</x:link>, to personalize content and analyze usage to improve our products. We do not perform interest-based advertising or use third party tracking cookies on m.twitch.tv. By clicking "Accept Cookies", you consent to this activity.',
            {
              'x:link': CookieLink,
            },
            'CookieNotice',
          )}
        </Layout>
        <Layout padding={{ top: 1 }}>
          <Button fullWidth onClick={acceptCookieNotice}>
            {formatMessage('Accept Cookies', 'CookieNotice')}
          </Button>
        </Layout>
      </ModalBody>
    </Modal>
  );
};

CookieNotice.displayName = 'CookieNotice';
